#ifndef __Device_CLASS__
#define __Device_CLASS__

class QString;

#include <fifo.h>
#include <value.h>
#include <list.h>
#include <opStack.h>

#define LOMEM   "low on memory"
#define ADDNAMED_OK	0
#define ADDNAMED_FULL	-1
#define ADDNAMED_OP	-2
#define ADDNAMED_UNIQUE	-3

#define CLK_RISING_EDGE		0
#define CLK_FALLING_EDGE	1
#define CLK_HIGH_VALUE		2
#define CLK_LOW_VALUE		3
#define CLK_NONE		4
#define CLK_MONO		5
#define CLK_MULTI		6

#define NOSPACE			1

// a logic device
class Device
{
public:
	Device(int , int delay=-1, int undef=-1, int clock=-1);
	~Device();

	//***** type and status methods
	int type();
	void changeLEDType(int);
	int delay();
	void setDelay(int);
	int getID();
	void setID(int);
	char * getText(int nospace=0);
	void setText(char *);
	int undef();
	void setUndef(int);
	int isInteractive();
	int sizeChangeable();
	int outputChanged();				// output changed since last access?

	//***** methods to manage named IO
	int hasNamedInput();
	int addInputName(const char *, int, int dev_id = 0);
	void changeInputName(int, const char *);
	void removeInputName(int);
	int getInputID(const char *);
	int inputNameIsUnique(const char *);

	int hasNamedOutput();
	int addOutputName(const char *, int, int dev_id = 0);
	void changeOutputName(int, const char *);
	void removeOutputName(int);
	char * getOutputName(int);
	int getOutputID(const char *);
	void setOutputNameType(int, int);

	int addInternalName(const char *);
	int isInternalName(int);
	void removeInternalName(int);

	list<value> *getNamedIRef();
	list<opStack> *getNamedORef();

	//***** oszillator properties
	int oszOff();
	void setOszOff(int);
	int oszOn();
	void setOszOn(int);

	//***** flipflop properties
	int hasMaster();
	void setMaster(int);

	//***** clock behaviour
	int clock();
	void setClock(int);
	int hasClock();

	//***** calculation methods
	void Calculate(int, int = 0);			// calculate device
	void flush(int);				// set all outputs to a new value

	//***** get values
	int input(int input_id = 0);
	int output(int output_id = 0);

	//***** set static input value (device must have the static operation OP_NONE)
	int hasStaticInput();
	void toggleStaticInput();
	void setStaticInput(int);			// set a static input value (e.g. a switch)
	void setStaticOutput(int, int);			// set a named output to a new value (used for net devices only)

	//***** equation related methods
	char * getEquation(int output_id = 0);		// return the equation for a specific output
	void setEquation();				// set equation string (for predefined devices)
	void setEquation(char);				// set equation string (for unnamed outputs)
	void setEquation(char *, int output_id = 0);	// set specific equation string (for named outputs)
	void setEquation(const char *, int output_id = 0);
	StackInfo parseEquation();			// setup operation stacks from the equation strings
	void resetParsing();				// clear all operation stacks
	// returns the correct name for an input
	char *qualifiedOutputName(Device *, char *, const char *qual_pref = (char *)NULL);
	// append all equations of this device to the given list
	void getAllEquations(list<OutputInfo> *, char *qualifier, int isolate = 0);

	//***** connection methods
	// connect two devices (1/0)
	int connectDevices(Device *, int, int);
	// disconnect two devices (1/0)
	int disconnectDevices(Device *, int, int);
	// exchange names of connections

	//***** device-import/export
	QString device2string();
	int string2device(QString);

	static int defUndefined();
	static void setDefUndefined(int);
	static int defDelay();
	static void setDefDelay(int);
	static int defClock();
	static void setDefClock(int);
	static void invertTrigger(int);
	static int triggerInverted();
	static void resetCounter();

	//***** device types
	//***** numbers 1000..1999 reserved (see ../include/mainw.h) 
	static const int fBASE = 1000;
	static const int fAND = 1000;
	static const int fOR = 1001;
	static const int fONE = 1002;
	static const int fXOR = 1003;
	static const int fSS = 1004;
	static const int fSWI = 1005;
	static const int fOSZ = 1006;
	static const int fINV_INTERNAL = 1007;
	static const int fRS = 1008;
	static const int fDFF = 1009;
	static const int fJK = 1010;
	static const int fLEDgreen = 1011;
	static const int fIN = 1012;
	static const int fOUT = 1013;
	static const int fNET = 1014;
	static const int fPWR = 1015;
	static const int fTXT = 1016;
	static const int fEQU = 1017;
	static const int fLEDred = 1018;
	static const int fLEDblue = 1019;
	static const int fLEDyellow = 1020;

	static const int MINDELAY=1;
	static const int MAXDELAY=99;

	static const int INPUT = 1;
	static const int TMP_OUTPUT = 2;
	static const int INTERNAL_OUTPUT = 3;
	static const int FINAL_OUTPUT = 4;

	static const int NO_QUALS = 1;

protected:
	// named output names/calculation stacks
	// id1: the device id of the interface device
	// id2: the number of the input of within this device
	list<opStack> named_output;

	// named input names/values
	// id1: the device id of the interface device
	// id2: the number of the input of within this device
	list<value> named_input;

	// id of probably used internal outputs within the device
	int mq_id;
	int c1_1_id;
	int s7_1_id;
	int s7_2_id;
	int s7_4_id;
	int s7_8_id;

private:
	static int STATcnt[20];

	static int STATdef_delay;
	static int STATdef_undefined;
	static int STATdef_clock;
	static int STATreverse_trigger;

	// calculate output value
	inline void calculate(int);

	// input-connection of device
	list<Device> connection_list;

	// function of the device
	int function;

	// static input value (devices without inputs, e.g. the switch)
	int static_input_value;

	// static output value (net device only)
	int static_output_value;

	// sum over all output values (needed for drawing)
	int old_result;
	int new_result;

	// oszillator timing
	int osz_off_cnt;
	int osz_on_cnt;
	int osz_curr_step;

	// pulsed devices
	int clock_type;
	int last_clk;

	// master flipflop enable/disable
	int master;

	// name of the device
	int id;

	// text label
	char *text;
// ***********************
// to be removed..
public:
	QString hist_string2device(const char *s);
// .. to be removed
// ***********************
};

#endif

