/***************************************************************************
                          setiloc.h  -  description
                             -------------------
    begin                : Wed Oct 27 1999
    copyright            : (C) 1999-2000 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SETILOC_H
#define SETILOC_H

#include <kprocess.h>
#include <kiconloader.h>

#include <qfile.h>
#include <qstring.h>
#include <qcolor.h>
#include <qlistview.h>
#include <qpixmap.h>

#include "main.h"

class SetiLoc : public QObject
{
Q_OBJECT

public:
	enum LocListView {AnalysisList, DataInfoList, UserInfoList};
  /**
   * Status definition values
   */
  enum SetiStatus {Stopped, Running, Finished, Loading};
	/** the constructor */
	SetiLoc(const QString& dir, const QString& loc, QColor col = QColor::QColor());
	/** the destructor */
	~SetiLoc();
  /**
   * Adds a QListView item to the list specified by list and lv.
   * @param list an integer defining the list to which the item shall be added.
   * Can be AnalysisList, DataInfoList, or UserInfoList
   * @param lv pointer to the QListView widget
   * @return a pointer to new ListView item
   */
  QListViewItem* addItem(int list, QListView *lv);
  /** Removes the QListView item from the list specified by list.
	 * @param list Can be AnalysisList, DataInfoList, or UserInfoList.
	 */
  void removeItem(int list);
	/**
	 * Removes all ListView items from the analysis list, data info list and
	 * user info list. Items are deleted.
	 */
	void removeAllItems();
	/** Returns the list view item of @e list. */
  inline QListViewItem* item(int list) const {return listitem[list];}
  /** Returns the location identifier string. */
  inline QString location() const {return slocation;}
  /** Sets the location identifier string. */
  inline void setLocation(const QString& l) {slocation = l;}
  /** Returns the location's directory. */
  inline QString directory() const {return sdirectory;}
  /** Sets the location's directory. */
  inline void setDirectory(const QString& d) {sdirectory = d;}
  /** Returns the location's color. */
  inline QColor color() const {return scolor;}
  /** Sets the location's color. */
  inline void setColor(QColor c) {scolor = c;}
  /** Returns the location's command line arguments. */
  inline QString cmdLineArgs() const {return scmdlineargs;}
  /** Sets the location's command line arguments. */
  inline void setCmdLineArgs(const QString& c)
  	{
  	scmdlineargs = c;
  	scmdlineargs = scmdlineargs.simplifyWhiteSpace();
  	}
  /** Returns the location's start-up flag. */
  inline bool startup() const {return sstartup;}
  /** Sets the location's start-up flag. */
  inline void setStartup(bool s) {sstartup = s;}
  /** Returns the location's Log-WU option */
  inline bool logwu() const {return slogwu;}
  /** Sets the location's Log-WU option */
  inline void setLogwu(bool s) {slogwu = s;}
  /** Returns the timestamp of the last logged WU. */
  QString timestamp() const;
  /** Sets the timestamp of the last logged WU. */
  void setTimestamp(const QString& ts);
  /** Returns the current WU name. */
  inline QString name() const {return wu_name;}
  /** Returns the date of orecord of the current WU. */
  inline QString rec_date() const {return recorded;}
  /** Returns the current RA value. */
  inline double ra() const {return sra;}
  /** Returns the current declination value. */
  inline double dec() const {return sdec;}
  /** Returns the number of WUs completed. */
  inline int results() const {return wu_completed;}
  /** Returns the total CPU time. */
  inline double totalCPUTime() const {return total_cpu;}
  /** Returns the CPU time consumed by the current WU. */
  inline double CPUTime() const {return cputime;}
  /** Returns the time left as a string. */
  inline QString timeLeftAsString() const {return(convertTime(timeleft, true));}
  /** Returns the state of the location. */
  inline int state() const {return status;}
  /** Returns the progress of the location. */
  inline double completed() const {return progress;}
  /** Sets the state of the location. */
  inline void setState(int st) {status = st; progress_old = progress;}
  /** Returns a pointer to a structure containing the top scores */
  inline WUScore* topScore() const {return (WUScore*)&max;}
  /** Sets a flag whether the Gaussian window is visible. */
  enum GraphIds {GaussianGraph, PulseGraph, TripletGraph};
  inline void setGraphVisible(int id, bool visible, QWidget* wid=0)
  	{
  	graphvsbl[id] = visible;
  	graphwidget[id] = (QWidget*)wid;
  	}
  /** Tells you whether the Gaussian window is visible. */
  inline bool isGraphVisible(int id) const {return graphvsbl[id];}
  /** Returns a pointer to the Gaussian widget. */
  inline QWidget* graphWidget(int id) const {return graphwidget[id];}
  /** Refreshes the entries of @e list. */
  bool refresh(int list);
  /**
   * Checks the status of a client.
   * @param state_f the state file string
   * @param data_f the work unit file string
   * @param result_f the result file string
   * @param resultheader_f the resultheader file string
   * @param wtemp_f the wtemp file string
   * @return the current status; 0 = Stopped, 1 = Running, 2 = Finished, 3 = Loading
   */
  int checkStatus(const QString& state_f, const QString& data_f, const QString& result_f,
  								const QString& resultheader_f, const QString& wtemp_f);
  /** Converts the time (as double value) to a string. */
	static QString convertTime(double time, bool hms);
	/** Reads an entry given by @e e from file @e fn. */
	static QString readEntry(const QString& fn, const QString& e);
  /**
	 * Checks the version of the running SETI@home client
	 * @return 0 when version couldn't be detected, 1 for version 1.0x,
	 *         2 for version 2.x, 3 for version 3.x, etc.
	 */
  int checkVersion();
  /** Displays the status icon in the Analysis list. */
  void showStatusIcon();
  /**
   * Starts the SETI@home client.
   * @return true if started successfully, otherwise false
   */
  bool startClient();
  /**
   * Stops the SETI@home client.
   * @return 0 if stopped successfully, otherwise -1
   */
  int stopClient();
  /**
   * Checks if the S@h client is running by trying to signal
   * the process pid.
   * @param pid the pid of the client. If omitted the pid will be
   * retrieved from pid.sah
   * @return true if client is running, false in case it is stopped. */
  bool isClientRunning(pid_t pid = 0);


protected:
	/** Reads the spike and Gaussian values from the file @e sf. */
	WUScore readStateData(const QString& sf) const;
	/** Reads the WU name from the file @e wuf. */
	QString readWUName(const QString& wuf) const;
	/** Reads the Gaussian data from the file @e sf. */
	bool readGaussianData(const QString& sf);
  /** Reads the pulse data from the file @e sf. */
  bool readPulseData(const QString& sf);
  /** Reads the triplet data from the file @e sf. */
  bool readTripletData(const QString& sf);
  /**
   * Retrieves the pid of the S@h client.
   * @return the pid, or -1 if pid couldn't be read
   */
  int getClientPid();
	
private:
	QString	slocation;
	QString	sdirectory;
	QColor	scolor;
	QString scmdlineargs;
	bool    sstartup;
	bool    slogwu;
	/** Time stamp of result file; prevents multiple logging. */
 	QString sresult_timestamp;
	QListViewItem *listitem[3];
	KIconLoader* statusIcons;
	QPixmap status_icon[4];
	/** Controls the SETI@home client */
	KProcess client;
	/** The process id of the S@h client, 0 if pid couldn't be obtained. */
	pid_t client_pid;
	// for analysis list
	int			status;
	int			statuscheckcount;
	double	chirprate;
	double	progress;
	double	progress_old;
	double	cputime;
	double	timeleft;
	double	totaltime;
	double	progressrate;
	double	mflops;
	WUScore max;
	bool    graphvsbl[3];
	QWidget* graphwidget[3];
	// for data info list
	QString from;
	QString wu_name;
	double	sra;
	double	sdec;
	QString recorded;
	QString source;
	double  base_frequency;
	// for user info list
	int			wu_requested;
	int			wu_completed;
	double	total_cpu;
	double	average_cpu;
	QString	last_result;
	QString	register_time;

signals:
	void newGaussian(SetiLoc* loc);
	void newPulse(SetiLoc* loc);
	void newTriplet(SetiLoc* loc);
	void newWorkUnit(SetiLoc* loc);
	void statusChanged();
};

#endif
