/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#ifndef CATALOGITEM_H
#define CATALOGITEM_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qstringlist.h>

class QTextStream;

/**
* This class represents an entry in a catalog.
* It contains the comment, the Msgid and the Msgstr.
* It defines some functions to query the state of the entry
* (fuzzy, untranslated, cformat).
*
* @short Class, representing an entry in a catalog
* @author Matthias Kiefer <matthias.kiefer@gmx.de>
*/
class CatalogItem
{

public:
    enum IOStatus{Ok,ParseError,Obsolete};
    /** Constructor */
    CatalogItem();
    /** Constructs a catalogitem from a string*/
    CatalogItem(QString item);
    /** Destructor */
    ~CatalogItem();

    /**
    * returns true, if the translation is fuzzy, this means
    * if the string ", fuzzy" is contained in the comment
    */
    bool isFuzzy() const;
    /**
    * returns true, if the translation in c-format, this means
    * if the string ", c-format" is contained in the comment
    */
    bool isCformat() const;
    /**
    * returns true, if the Msgid is untranslated, this means the
    * Msgstr is empty
    */
    bool isUntranslated() const;

    bool isValid() const {return _valid;}

    /** returns the number of lines, the entry will need in a file */
    int totalLines() const;

    /** removes the string ", fuzzy" from the comment */
    void removeFuzzy();

    /** cleares the item */
    void clear();


    /** returns the comment of this entry */
    QString comment() const{return _comment;}
    /** returns the msgid of the entry */
    QString msgid() const{return _msgid;}
    /** returns the msgstr of the entry */
    QString msgstr() const{return _msgstr;}

    QStringList msgidAsList() const;
    QStringList msgstrAsList() const;
    QStringList commentAsList() const;

    void setComment(QString com){_comment=com;}
    void setMsgid(QString msg){_msgid=msg;}
    void setMsgstr(QString msg){_msgstr=msg;}


    /** returns the complete entry as a string */
    QString asString();

    /**
    * reads an entry from the stream
    * @see QTextStream
    */
    IOStatus read(QTextStream& stream);
    /**
    * writes an entry to the stream
    * @see QTextStream
    */
    void write(QTextStream& stream) const;

private:
   QString _comment;
   QString _msgid;
   QString _msgstr;

   bool _valid;

};

#endif // CATALOGITEM_H
