/*
    KMLOVoice

    A utility to process voice messages received with the ELSA
    MicroLink(tm) Office modem.

    Copyright (C) 2000 Oliver Gantz <Oliver.Gantz@epost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ------
    ELSA and MicroLink are trademarks of ELSA AG, Aachen.
*/

#include <qglobal.h>
#include <qframe.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qsize.h>
#include <qtooltip.h>
#include <qfile.h>

#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ksimpleconfig.h>
#include <kseparator.h>

#include "senderaliases.h"
#include "global.h"



CallerAliases *calleraliases = 0;



EditAlias::EditAlias(QWidget *parent, const char *name) : KDialog (parent, name, true)
{
	QVBoxLayout *vbox;
	QHBoxLayout *hbox;
	QGridLayout *grid;
	QLabel *label;
	KSeparator *separator;
	QPushButton *ok, *cancel;
	QSize size;

	vbox = new QVBoxLayout(this, marginHint());
	CHECK_PTR(vbox);

	vbox->addStretch();

	grid = new QGridLayout(2, 2);
	CHECK_PTR(grid);
	vbox->addLayout(grid);

	label = new QLabel(i18n("&Caller ID:"), this);
	CHECK_PTR(label);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 0, AlignVCenter | AlignLeft);

	m_caller = new QLineEdit(this);
	CHECK_PTR(m_caller);
	label->setBuddy(m_caller);
	m_caller->setMinimumWidth(200);
	m_caller->setFixedHeight(m_caller->sizeHint().height());
	grid->addWidget(m_caller, 0, 1);

	label = new QLabel(i18n("&Alias:"), this);
	CHECK_PTR(label);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0, AlignVCenter | AlignLeft);

	m_alias = new QLineEdit(this);
	CHECK_PTR(m_alias);
	label->setBuddy(m_alias);
	m_alias->setMinimumWidth(200);
	m_alias->setFixedHeight(m_alias->sizeHint().height());
	grid->addWidget(m_alias, 1, 1);

	vbox->addStretch();
	
	vbox->addSpacing(marginHint());

	separator = new KSeparator(KSeparator::HLine, this);
	CHECK_PTR(separator);
	vbox->addWidget(separator);
	
	hbox = new QHBoxLayout();
	CHECK_PTR(hbox);
	vbox->addLayout(hbox);

	ok = new QPushButton(i18n("&OK"), this);
	CHECK_PTR(ok);
	ok->setDefault(true);
	cancel = new QPushButton(i18n("&Cancel"), this);
	CHECK_PTR(cancel);

	size = ok->sizeHint().expandedTo(cancel->sizeHint());

	ok->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addStretch();
	hbox->addWidget(ok);
	hbox->addWidget(cancel);

	vbox->activate();

	connect(ok, SIGNAL(clicked()), SLOT(accept()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	setCaption(i18n("Caller Alias"));
}


EditAlias::~EditAlias()
{
}



CallerAliases::CallerAliases(QWidget *parent, const char *name) : KDialog (parent, name, false)
{
	QVBoxLayout *vbox, *gvbox;
	QHBoxLayout *hbox, *ghbox;
	QGroupBox *gb;
	QSize size;
	QPushButton *bhelp, *bclose;

	gotToolTips = false;

	vbox = new QVBoxLayout(this, marginHint());
	CHECK_PTR(vbox);

	gb = new QGroupBox(this);
	CHECK_PTR(gb);

	gvbox = new QVBoxLayout(gb, marginHint());
	CHECK_PTR(gvbox);
	
	list = new QListView(gb, "aliaslist");
	CHECK_PTR(list);
	list->addColumn(i18n("Caller ID"));
	list->addColumn(i18n("Alias"));
	list->setColumnWidth(0, 200);
	list->setColumnWidth(1, 280);
	list->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	list->setAllColumnsShowFocus(true);
	list->setShowSortIndicator(true);

	gvbox->addWidget(list, 1);

	ghbox = new QHBoxLayout();
	CHECK_PTR(ghbox);
	gvbox->addLayout(ghbox);

	add_item = new QPushButton(i18n("&Add"), gb);
	CHECK_PTR(add_item);
	edit_item = new QPushButton(i18n("&Edit"), gb);
	CHECK_PTR(edit_item);
	edit_item->setEnabled(false);
	del_item = new QPushButton(i18n("&Delete"), gb);
	CHECK_PTR(del_item);
	del_item->setEnabled(false);

	size = add_item->sizeHint().expandedTo(edit_item->sizeHint()).expandedTo(del_item->sizeHint());

	add_item->setFixedSize(size);
	edit_item->setFixedSize(size);
	del_item->setFixedSize(size);

	ghbox->addWidget(add_item);
	ghbox->addStretch();
	ghbox->addWidget(edit_item);
	ghbox->addStretch();
	ghbox->addWidget(del_item);

	gvbox->activate();

	vbox->addWidget(gb, 1);
	
	hbox = new QHBoxLayout(8);
	CHECK_PTR(hbox);
	vbox->addLayout(hbox);

	bhelp = new QPushButton(i18n("&Help"), this);
	CHECK_PTR(bhelp);
	bclose = new QPushButton(i18n("&Close"), this);
	CHECK_PTR(bclose);
	bclose->setDefault(true);
	size = bhelp->sizeHint().expandedTo(bclose->sizeHint());
	bhelp->setFixedSize(size);
	bclose->setFixedSize(size);

	hbox->addWidget(bhelp);
	hbox->addStretch();
	hbox->addWidget(bclose);

	vbox->activate();

	connect(list, SIGNAL(selectionChanged(QListViewItem *)), SLOT(selectionChangedSlot(QListViewItem *)));
	connect(list, SIGNAL(doubleClicked(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(list, SIGNAL(returnPressed(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(add_item, SIGNAL(clicked()), SLOT(addItem()));
	connect(edit_item, SIGNAL(clicked()), SLOT(editItem()));
	connect(del_item, SIGNAL(clicked()), SLOT(deleteItem()));
	connect(bhelp, SIGNAL(clicked()), SLOT(slotHelp()));
	connect(bclose, SIGNAL(clicked()), SLOT(close()));

	setCaption(i18n("Caller Aliases"));

	loadDir();
}


CallerAliases::~CallerAliases()
{
}


void CallerAliases::loadDir()
{
	KSimpleConfig *sconfig;
	int number, i;
	QListViewItem *item;
	
	list->clear();

	sconfig = new KSimpleConfig(expandPath(KMLOVOICE_ALIASLIST), true);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries");
	for (i = 0; i < number; i++) {
		sconfig->setGroup(QString("Entry %1").arg(i));
		item = new QListViewItem(list);
		CHECK_PTR(item);
		item->setText(0, sconfig->readEntry("Caller"));
		item->setText(1, sconfig->readEntry("Alias"));
	}
	delete sconfig;
}


void CallerAliases::saveDir()
{
	KSimpleConfig *sconfig;
	QListViewItem *item;
	int i;

	QFile::remove(expandPath(KMLOVOICE_ALIASLIST));
	sconfig = new KSimpleConfig(expandPath(KMLOVOICE_ALIASLIST), false);
	CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	sconfig->writeEntry("Entries", list->childCount());
	item = list->firstChild();
	for (i = 0; (item); i++) {
		sconfig->setGroup(QString("Entry %1").arg(i));
		sconfig->writeEntry("Caller", item->text(0));
		sconfig->writeEntry("Alias", item->text(1));
		item = item->nextSibling();
	}
	sconfig->sync();
	delete sconfig;
}


void CallerAliases::addToolTips(bool on)
{
	if (on == gotToolTips)
		return;

	gotToolTips = on;

	if (gotToolTips) {
		QToolTip::add(add_item, i18n("Add an entry to the list"));
		QToolTip::add(edit_item, i18n("Edit selected entry"));
		QToolTip::add(del_item, i18n("Delete selected entry"));
	}
	else {
		QToolTip::remove(add_item);
		QToolTip::remove(edit_item);
		QToolTip::remove(del_item);
	}
}	


QString CallerAliases::alias(const QString& id_)
{
	QListViewItem *item;
	QString s = QString(id_.stripWhiteSpace());

	if (s.isEmpty())
		return i18n("<unknown>");
	
	item = list->firstChild();
	while (item) {
		if (item->text(0) == s)
			return item->text(1);
		item = item->nextSibling();
	}

	return id_;
}


void CallerAliases::setAlias(const QString& id_)
{
	QListViewItem *item;
	QString s = QString(id_.stripWhiteSpace());

	item = list->firstChild();
	while (item) {
		if (item->text(0) == s)
			break;
		item = item->nextSibling();
	}

	if (!item) {
		item = new QListViewItem(list);
		item->setText(0, s);
	}

	list->setSelected(item, true);

	if (isVisible())
		raise();
	else
		show();
}


void CallerAliases::selectionChangedSlot(QListViewItem *item)
{
	if (item) {
		edit_item->setEnabled(true);
		del_item->setEnabled(true);
	}
}


void CallerAliases::executedSlot(QListViewItem *)
{
	editItem();
}


void CallerAliases::addItem()
{
	EditAlias edit;
	QListViewItem *item;

	if (edit.exec()) {
		item = new QListViewItem(list);
		item->setText(0, edit.caller());
		item->setText(1, edit.alias());
		list->setSelected(item, true);
		list->ensureItemVisible(item);
		saveDir();
		emit applyButtonPressed();
	}
}


void CallerAliases::editItem()
{
	EditAlias edit;
	QListViewItem *item;

	if ((item = list->currentItem())) {
		edit.setCaller(item->text(0));
		edit.setAlias(item->text(1));
		if (edit.exec()) {
			item->setText(0, edit.caller());
			item->setText(1, edit.alias());
			saveDir();
			emit applyButtonPressed();
		}
	}
}


void CallerAliases::deleteItem()
{
	QListViewItem *item;

	if ((item = list->currentItem())) {
		if (KMessageBox::questionYesNo(this, i18n("Really delete entry\n\"%1\"?").arg(item->text(0)), i18n("Question"), i18n("&Delete"), i18n("&Cancel")) == KMessageBox::Yes) {
			delete item;
			edit_item->setEnabled(false);
			del_item->setEnabled(false);
			saveDir();
			emit applyButtonPressed();
		}
	}
}


void CallerAliases::slotHelp()
{
	kapp->invokeHelp("CALLERALIASES-HELP");
}
