/**
 * kconfigparser.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "kconfigparser.h"
#include "entrytoken.h"
#include "grouptoken.h"
#include "applicationtoken.h"
#include "hosttoken.h"
#include "hostdata.h"

#include <kconfig.h>
#include <ktempfile.h>
#include <kdebug.h>

#include <qfileinfo.h>
#include <qmap.h>
#include <qptrlist.h>

namespace KCfgCreator
{

KConfigParser::KConfigParser( QObject* parent )
    : ParserInterface( parent )
{
}

ApplicationToken*
KConfigParser::parseAndAddFile( HostToken* parent, FileData* fdata )
{
    if ( skipFile( fdata->url().fileName() ) )
        return 0;


    ApplicationToken* token = parent->application( fdata->url() );
    if ( !token ) {
        token = new ApplicationToken( fdata->url(), fdata->url().fileName() );
        parent->addApp( token );
    }

    KURL fileURL;

    if ( !fdata->url().isLocalFile() ) {
        KTempFile tempFile;
        tempFile.setAutoDelete( true );
        tempFile.dataStream()->writeBytes( fdata->data,
                                           fdata->data.size() );
        tempFile.close();
        fileURL.setPath( tempFile.name() );
        token = parseFileUrl( fileURL, token );
    } else {
        fileURL = fdata->url();
        token = parseFileUrl( fileURL, token );
    }

    return token;
}

ApplicationToken*
KConfigParser::parseFileUrl( const KURL& url, ApplicationToken *token )
{
    KConfig conf( url.path(), false, false );

    QStringList list( conf.groupList() );

    if ( !token )
        token = new ApplicationToken( url, url.fileName() );

    for ( QStringList::iterator it = list.begin(); it != list.end(); ++it ) {
        parseGroup( token, conf, *it );
    }
    return token;
}

void
KConfigParser::parseData( HostData* data )
{
    Q_ASSERT( data );
    HostToken *hostToken = new HostToken( data->url(), data->url().host() );

    QPtrList<FileData> fileData = data->kconfigFiles();
    QPtrListIterator<FileData> itr( fileData );

    while ( itr.current() ) {
        if ( skipFile( itr.current()->url().fileName() ) ) {
            ++itr;
            continue;
        }

        KURL fileURL;
        if ( !itr.current()->url().isLocalFile() ) {
            KTempFile tempFile;
            tempFile.setAutoDelete( true );
            *tempFile.dataStream() << itr.current()->data;
            tempFile.close();
            fileURL.setPath( tempFile.name() );
        } else {
            fileURL = itr.current()->url();
        }

        ApplicationToken* token = new ApplicationToken( itr.current()->url(),
                                                        itr.current()->url().fileName() );
        if ( token ) {
            parseFileUrl( fileURL, token );
            hostToken->addApp( token ) ;
        }
        ++itr;
    }
    emit parsed( hostToken );
}

void
KConfigParser::parseGroup( ApplicationToken* parent, const KConfig& config, const QString& group )
{
    QMap<QString, QString> en = config.entryMap( group );

    GroupToken* token = parent->group( group );
    if ( !token )
        token = parent->createGroup( group, true );

    parseEntries( token, config, en );
}

void
KConfigParser::parseEntries( GroupToken* token, const KConfig& /*config*/,
                             const QMap<QString, QString>& en )
{
    //FIXME: immutable, defaults... etc.
    QMap<QString, QString>::ConstIterator it;
    for ( it = en.begin(); it != en.end(); ++it ) {
        EntryToken* etoken = token->entry( it.key() );
        if ( !etoken )
            etoken = token->createEntry( it.key(), "String" );
        etoken->setValueFromString( it.data() );
    }
}

ApplicationToken*
KConfigParser::parseDataMerging( ApplicationToken* app,
                                 FileData* fdata )
{
    if ( skipFile( fdata->url().fileName() ) ) {
        return app;
    }

    KURL fileURL;
    if ( !fdata->url().isLocalFile() ) {
        KTempFile tempFile;
        tempFile.setAutoDelete( true );
        *tempFile.dataStream() << fdata->data;
        tempFile.close();
        fileURL.setPath( tempFile.name() );
    } else {
        fileURL = fdata->url();
    }

    return parseFileUrl( fileURL, app );
}

bool
KConfigParser::skipFile( const QString& fileName )
{
    if ( fileName == "kdebug.areas" )
        return true;
    else if ( fileName == "kxkb_groups" )
        return true;
    else if ( fileName == "gtkrc" )
        return true;

    return false;
}

}

#include "kconfigparser.moc"
