/**
 * entries.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "entries.h"

using namespace KConfigEditor;


StringEntry::StringEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<QString>( parent, name, hidden )
{
}

QVariant StringEntry::variant() const
{
    return QVariant( m_value );
}

void StringEntry::setVariant( const QVariant& var )
{
    m_value = var.toString();

    finishSetting();
}

QString  StringEntry::valueAsString() const
{
    return m_value;
}

QString StringEntry::initialValueAsString() const
{
    return m_initialValue;
}

void StringEntry::setValueFromString( const QString& value )
{
    m_value = value;

    finishSetting();
}

StringListEntry::StringListEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<QStringList>( parent, name, hidden )
{
}

QVariant StringListEntry::variant() const
{
    return QVariant( m_value );
}

void StringListEntry::setVariant( const QVariant& var )
{
    m_value = var.toStringList();

    finishSetting();
}

QString  StringListEntry::valueAsString() const
{
    return m_value.join( ", " );
}

QString StringListEntry::initialValueAsString() const
{
    return m_initialValue.join( ", " );
}

void StringListEntry::setValueFromString( const QString& value )
{
    m_value = QStringList::split( value,  ", " );

    finishSetting();
}

FontEntry::FontEntry( GroupToken* parent, const QString& name , bool hidden )
    : EntriesBase<QFont>( parent, name, hidden )
{
}

QVariant FontEntry::variant() const
{
    return QVariant( m_value );
}

void FontEntry::setVariant( const QVariant& var )
{
    m_value = var.toFont();
    finishSetting();
}

QString  FontEntry::valueAsString() const
{
    return m_value.toString();
}

QString  FontEntry::initialValueAsString() const
{
    return m_initialValue.toString();
}

void FontEntry::setValueFromString( const QString& val )
{
    m_value.fromString( val );

    finishSetting();
}

RectEntry::RectEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<QRect>( parent, name, hidden )
{
}

QVariant RectEntry::variant() const
{
    return QVariant( m_value );
}

void RectEntry::setVariant( const QVariant& var )
{
    m_value = var.toRect();
    finishSetting();
}

QString  RectEntry::valueAsString() const
{
    return QString( "%1,%2,%3,%4" )
        .arg( m_value.left() )
        .arg( m_value.top() )
        .arg( m_value.width() )
        .arg( m_value.height() );
}

QString  RectEntry::initialValueAsString() const
{
    return QString( "%1,%2,%3,%4" )
        .arg( m_initialValue.left() )
        .arg( m_initialValue.top() )
        .arg( m_initialValue.width() )
        .arg( m_initialValue.height() );
}


void RectEntry::setValueFromString( const QString& val )
{
    if ( !val.isEmpty() ) {
        int left, top, width, height;
        if ( sscanf( val.latin1(), "%d,%d,%d,%d", &left, &top,
                     &width, &height ) == 4 ) {
            m_value = QRect( left, top, width, height );
            finishSetting();
            return;
        }
    }

    //rectangle coordinates are not in the passed
    //string (val)
    m_value = QRect();
    finishSetting();
}

SizeEntry::SizeEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<QSize>( parent, name, hidden )
{
}

QVariant SizeEntry::variant() const
{
    return QVariant( m_value );
}

void SizeEntry::setVariant( const QVariant& var )
{
    m_value = var.toSize();
    finishSetting();
}

QString  SizeEntry::valueAsString() const
{
    return QString( "%1,%2" )
        .arg( m_value.width() )
        .arg( m_value.height() );
}

QString  SizeEntry::initialValueAsString() const
{
    return QString( "%1,%2" )
        .arg( m_initialValue.width() )
        .arg( m_initialValue.height() );
}

void SizeEntry::setValueFromString( const QString& val )
{
    if ( !val.isEmpty() )
    {
        int width, height;
        if ( sscanf( val.latin1(), "%d,%d", &width, &height ) == 2 )
        {
            m_value = QSize( width, height );
            finishSetting();
            return;
        }
    }

    m_value = QSize();
    finishSetting();
}


PointEntry::PointEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<QPoint>( parent, name, hidden )
{
}

QVariant PointEntry::variant() const
{
    return QVariant( m_value );
}

void PointEntry::setVariant( const QVariant& var )
{
    m_value = var.toPoint();
    finishSetting();
}

QString  PointEntry::valueAsString() const
{
    return QString( "%1,%2" )
        .arg( m_value.x() )
        .arg( m_value.y() );
}

QString  PointEntry::initialValueAsString() const
{
    return QString( "%1,%2" )
        .arg( m_initialValue.x() )
        .arg( m_initialValue.y() );
}

void PointEntry::setValueFromString( const QString& val )
{
    if ( !val.isEmpty() )
    {
        int x, y;
        if ( sscanf( val.latin1(), "%d,%d", &x, &y ) == 2 )
        {
            m_value = QPoint( x, y );
            finishSetting();
            return;
        }
    }
    //if got here val wasn't a valid point
    m_value = QPoint();
    finishSetting();
}

ColorEntry::ColorEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<QColor>( parent, name, hidden )
{
}

QVariant ColorEntry::variant() const
{
    return QVariant( m_value );
}

void ColorEntry::setVariant( const QVariant& var )
{
    m_value = var.toColor();
    finishSetting();
}

QString  ColorEntry::valueAsString() const
{
    return m_value.name();
}

QString  ColorEntry::initialValueAsString() const
{
    return m_initialValue.name();
}

void ColorEntry::setValueFromString( const QString& val )
{
    m_value.setNamedColor( val );
    finishSetting();
}

IntEntry::IntEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<int>( parent, name, hidden )
{
}

QVariant IntEntry::variant() const
{
    return QVariant( m_value );
}

void IntEntry::setVariant( const QVariant& var )
{
    m_value = var.toInt();
    finishSetting();
}

QString  IntEntry::valueAsString() const
{
    return QString::number( m_value );
}

QString  IntEntry::initialValueAsString() const
{
    return QString::number( m_initialValue );
}

void IntEntry::setValueFromString( const QString& val )
{
    m_value = val.toInt();
    finishSetting();
}

UIntEntry::UIntEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<uint>( parent, name, hidden )
{
}

QVariant UIntEntry::variant() const
{
    return QVariant( m_value );
}

void UIntEntry::setVariant( const QVariant& var )
{
    m_value = var.toUInt();
    finishSetting();
}

QString  UIntEntry::valueAsString() const
{
    return QString::number( m_value );
}

QString  UIntEntry::initialValueAsString() const
{
    return QString::number( m_initialValue );
}

void UIntEntry::setValueFromString( const QString& val )
{
    m_value = val.toUInt();
    finishSetting();
}

BoolEntry::BoolEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<bool>( parent, name, hidden )
{
}

QVariant BoolEntry::variant() const
{
    return QVariant( m_value );
}

void BoolEntry::setVariant( const QVariant& var )
{
    m_value = var.toBool();
    finishSetting();
}

QString  BoolEntry::valueAsString() const
{
    return m_value ? "true" : "false";
}

QString  BoolEntry::initialValueAsString() const
{
    return m_initialValue ? "true" : "false";
}

void BoolEntry::setValueFromString( const QString& val )
{
    if ( val == "true" ) m_value = true;
    else m_value = false;
    finishSetting();
}

DoubleEntry::DoubleEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<double>( parent, name, hidden )
{
}

QVariant DoubleEntry::variant() const
{
    return QVariant( m_value );
}

void DoubleEntry::setVariant( const QVariant& var )
{
    m_value = var.toDouble();
    finishSetting();
}

QString  DoubleEntry::valueAsString() const
{
    return QString::number( m_value );
}

QString  DoubleEntry::initialValueAsString() const
{
    return QString::number( m_initialValue );
}

void DoubleEntry::setValueFromString( const QString& val )
{
    m_value = val.toDouble();
    finishSetting();
}

DateTimeEntry::DateTimeEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<QDateTime>( parent, name, hidden )
{
}

QVariant DateTimeEntry::variant() const
{
    return QVariant( m_value );
}

void DateTimeEntry::setVariant( const QVariant& var )
{
    m_value = var.toDateTime();
    finishSetting();
}

QString  DateTimeEntry::valueAsString() const
{
    return m_value.toString();
}

QString  DateTimeEntry::initialValueAsString() const
{
    return m_initialValue.toString();
}

void DateTimeEntry::setValueFromString( const QString& val )
{
    m_value = QDateTime::fromString( val );
    finishSetting();
}

Int64Entry::Int64Entry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<Q_LLONG>( parent, name, hidden )
{
}

QVariant Int64Entry::variant() const
{
    return QVariant( m_value );
}

void Int64Entry::setVariant( const QVariant& var )
{
    m_value = var.toLongLong();
    finishSetting();
}

QString  Int64Entry::valueAsString() const
{
    return QString::number( m_value );
}

QString  Int64Entry::initialValueAsString() const
{
    return QString::number( m_initialValue );
}

void Int64Entry::setValueFromString( const QString& val )
{
    m_value = val.toLongLong();
    finishSetting();
}

UInt64Entry::UInt64Entry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<Q_ULLONG>( parent, name, hidden )
{
}

QVariant UInt64Entry::variant() const
{
    return QVariant( m_value );
}

void UInt64Entry::setVariant( const QVariant& var )
{
    m_value = var.toULongLong();
    finishSetting();
}

QString  UInt64Entry::valueAsString() const
{
    return QString::number( m_value );
}

QString  UInt64Entry::initialValueAsString() const
{
    return QString::number( m_initialValue );
}

void UInt64Entry::setValueFromString( const QString& val )
{
    m_value = val.toULongLong();
    finishSetting();
}

IntListEntry::IntListEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase< QValueList<int> >( parent, name, hidden )
{
}

QVariant IntListEntry::variant() const
{
    return QVariant( "FIXME" );
}

void IntListEntry::setVariant( const QVariant&  )
{
    //FIXME implement
    //m_value = var.toList();
    //finishSetting();
}

QString  IntListEntry::valueAsString() const
{
    QStringList list;
    for ( unsigned int i=0; i<m_value.count(); ++i )
        list.append( QString( "%1" ).arg( m_value[ i ] ) );
    return list.join( "," );
}

QString  IntListEntry::initialValueAsString() const
{
    QStringList list;
    for ( unsigned int i=0; i<m_initialValue.count(); ++i )
        list.append( QString( "%1" ).arg( m_initialValue[ i ] ) );
    return list.join( "," );
}

void IntListEntry::setValueFromString( const QString& val )
{
    QStringList list = QStringList::split( ',', val );
    m_value.clear();
    for ( unsigned int i=0; i<list.count(); ++i )
        m_value.append( list[ i ].toInt() );
    finishSetting();
}

QVariant IntListEntry::initialValue() const
{
    return QVariant( "FIXME" );
}

EnumEntry::EnumEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase< QValueList<int> >( parent, name, hidden )
{
}

QVariant EnumEntry::variant() const
{
    return QVariant( "FIXME" );
}

void EnumEntry::setVariant( const QVariant& )
{
    //FIXME
    //finishSetting();
}

QString  EnumEntry::valueAsString() const
{
    return "FIXME";
}

QString  EnumEntry::initialValueAsString() const
{
    return "FIXME";
}

void EnumEntry::setValueFromString( const QString& )
{
    //FIXME
    //finishSetting();
}

QVariant EnumEntry::initialValue() const
{
    return QVariant( "FIXME" );
}

PathEntry::PathEntry( GroupToken* parent, const QString& name, bool hidden )
    : EntriesBase<KURL>( parent, name, hidden )
{
}

QVariant PathEntry::variant() const
{
    return QVariant( m_value.prettyURL() );
}

void PathEntry::setVariant( const QVariant& var )
{
    m_value = KURL( var.toString() );
    finishSetting();
}

QString  PathEntry::valueAsString() const
{
    return m_value.prettyURL();
}

QString  PathEntry::initialValueAsString() const
{
    return m_initialValue.prettyURL();
}

void PathEntry::setValueFromString( const QString& val )
{
    m_value = KURL();
    m_value.setPath( val );
    finishSetting();
}

QVariant PathEntry::initialValue() const
{
    return QVariant( m_initialValue.url() );
}
