/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <qmap.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qsize.h>
#include <qdir.h>
#include <qfileinfo.h>

#include <kseparator.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <ksimpleconfig.h>
#include <kmessagebox.h>

#include "folderview.h"
#include "faxview.h"
#include "global.h"


QPopupMenu *moveMenu = 0;
QPopupMenu *copyMenu = 0;

static QMap<int, FolderViewItem *> folderMap;


EditFolder::EditFolder(QWidget *parent, const char *name): KDialog (parent, name, true)
{
	QVBoxLayout *vbox;
	QHBoxLayout *hbox;
	QLabel *label;
	KSeparator *separator;
	QPushButton *ok, *cancel;
	QSize size;

	vbox = new QVBoxLayout(this, marginHint());
	Q_CHECK_PTR(vbox);

	vbox->addStretch();

	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	label = new QLabel(i18n("&Name:"), this);
	label->setMinimumSize(label->sizeHint());
	hbox->addWidget(label);

	m_name = new QLineEdit(this);
	label->setBuddy(m_name);
	m_name->setMinimumWidth(200);
	m_name->setFixedHeight(m_name->sizeHint().height());
	hbox->addWidget(m_name);

	vbox->addStretch();
	
	vbox->addSpacing(marginHint());

	separator = new KSeparator(KSeparator::HLine, this);
	vbox->addWidget(separator);
	
	hbox = new QHBoxLayout();
	vbox->addLayout(hbox);

	ok = new QPushButton(i18n("&OK"), this);
	ok->setDefault(true);
	cancel = new QPushButton(i18n("&Cancel"), this);

	size = ok->sizeHint().expandedTo(cancel->sizeHint());

	ok->setFixedSize(size);
	cancel->setFixedSize(size);

	hbox->addStretch();
	hbox->addWidget(ok);
//	hbox->addSpacing(spacingHint());
	hbox->addWidget(cancel);

	vbox->activate();

	connect(ok, SIGNAL(clicked()), SLOT(accept()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));

	m_name->setFocus();
}


EditFolder::~EditFolder()
{
}



FolderViewItem::FolderViewItem(QListView *parent, Type type, const char *name): QListViewItem(parent, name)
{
	init(type);
}


FolderViewItem::FolderViewItem(QListViewItem *parent, Type type, const char *name): QListViewItem(parent, name)
{
	init(type);
}


FolderViewItem::~FolderViewItem()
{
}


void FolderViewItem::init(Type type)
{
	m_type = type;

	if (m_type == Root) {
		setPixmap(0, SmallIcon("kmlofax"));
		setText(0, i18n("Facsimiles"));
	}
	else if (m_type == Inbox) {
		setPixmap(0, SmallIcon("inbox"));
		setText(0, i18n("Inbox"));
	}
	else if (m_type == Trash) {
		setText(0, i18n("Trash"));
	}
	else if (m_type == User) {
		setPixmap(0, SmallIcon("folder"));
	}
	setFiles(0);
}


void FolderViewItem::setName(const QString &name_)
{
	if (m_type == User)
		setText(0, name_);
}


QString FolderViewItem::name()
{
	return text(0);
}


QString FolderViewItem::path()
{
	if (m_type == Root)
		return FOLDERS_DIR;
	else if (m_type == Inbox)
		return INBOX_DIR;
	else if (m_type == Trash)
		return TRASH_DIR;

	return ((FolderViewItem *)parent())->path() + "/" + name();
}


void FolderViewItem::setFiles(int files_)
{
	QString s;
	
	m_files = files_;

	if (m_type == Root)
		return;

	setText(1, s.setNum(m_files));
	if (m_type == Trash)
		setPixmap(0, (files_) ? SmallIcon("trashcan_full") : SmallIcon("trashcan_empty"));
}

	
void FolderViewItem::markCurrent(bool current)
{
	if (m_type == Inbox)
		setPixmap(0, current ? SmallIcon("inbox_open") : SmallIcon("inbox"));
	else if (m_type == User)
		setPixmap(0, current ? SmallIcon("folder_open") : SmallIcon("folder"));
}


FolderView::FolderView(QWidget *parent, const char *name): KListView(parent, name)
{
	moveMenu = new QPopupMenu();
	copyMenu = new QPopupMenu();

	config = kapp->config();

	inbox_folder = 0;
	trash_folder = 0;
	current_folder = 0;

	dragging = false;
	old_item = 0;

	addColumn(i18n("Folder"), 0);
	addColumn(i18n("Items"), 0);
	setColumnAlignment(1, AlignVCenter | AlignRight);
	setMinimumSize(sizeHint());
	setSorting(-1);
	setAllColumnsShowFocus(true);

	setAcceptDrops(true);

	context_menu = new QPopupMenu(this, "foldercontext");
	Q_CHECK_PTR(context_menu);
	context_menu->insertItem(SmallIcon("folder_new"), i18n("&New Folder..."), ID_FOLDER_NEW);
	context_menu->insertItem(i18n("&Rename..."), ID_FOLDER_RENAME);
	context_menu->insertSeparator();
	context_menu->insertItem(i18n("&Empty"), ID_FOLDER_EMPTY);
	context_menu->insertItem(SmallIcon("remove"), i18n("Re&move"), ID_FOLDER_REMOVE);
	
	connect(this, SIGNAL(currentChanged(QListViewItem *)), SLOT(currentChangedSlot(QListViewItem *)));
	connect(this, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint&, int)), SLOT(rightButtonPressedSlot(QListViewItem *, const QPoint&, int)));
	connect(context_menu, SIGNAL(activated(int)), SLOT(activatedSlot(int)));

	init();

	readOptions();
}


FolderView::~FolderView()
{
	delete moveMenu;
	moveMenu = 0;
	delete copyMenu;
	copyMenu = 0;
}


void FolderView::readOptions()
{
	config->setGroup("FolderWindow Options");
	
	setColumnWidth(0, config->readNumEntry("FolderWidth", 100));
	setColumnWidth(1, config->readNumEntry("ItemsWidth", 48));
}


void FolderView::saveOptions()
{
	config->setGroup("FolderWindow Options");
	
	config->writeEntry("FolderWidth", columnWidth(0));
	config->writeEntry("ItemsWidth", columnWidth(1));
}


void FolderView::setFiles(FolderViewItem *item)
{
	KSimpleConfig *sconfig;
	int number;

	sconfig = new KSimpleConfig(expandPath(item->path() + "/" + KMLOFAX_FAXDIR), true);
	Q_CHECK_PTR(sconfig);
	sconfig->setGroup("General");
	number = sconfig->readNumEntry("Entries", 0);
	delete sconfig;
	
	item->setFiles(number);
}


void FolderView::init()
{
	FolderViewItem *root_item = new FolderViewItem(this, FolderViewItem::Root);

	QDir d(expandPath(FOLDERS_DIR));
	if (!d.exists())
		d.mkdir(expandPath(FOLDERS_DIR));

	init(root_item, expandPath(FOLDERS_DIR));

	trash_folder = new FolderViewItem(root_item, FolderViewItem::Trash);
	inbox_folder = new FolderViewItem(root_item, FolderViewItem::Inbox);

	root_item->setOpen(true);

	QListViewItemIterator it(firstChild());
	while (it.current()) {
		setFiles((FolderViewItem *)it.current());
		it++;
	}

	rebuildFolderMenus();
}


void FolderView::init(FolderViewItem *parent, const QString &dir)
{
	QDir d(dir);
	const QFileInfoList *files;
	FolderViewItem *item;

	d.setFilter(QDir::Dirs | QDir::NoSymLinks);
	files = d.entryInfoList();

	if (files) {
		QFileInfoListIterator it(*files);
		QFileInfo *f;
		while ((f = it.current()) != 0) {
			++it;
			if (f->fileName().at(0) != '.') {
				item = new FolderViewItem(parent, FolderViewItem::User);
				item->setName(f->fileName());
				init(item, f->filePath());
			}
		}
	}
}


QPopupMenu *FolderView::rebuildFolderMenu(FolderViewItem *item, QPopupMenu *menu_, bool move)
{
	int id;

	if (move)
		connect(menu_, SIGNAL(activated(int)), SLOT(moveSlot(int)));
	else
		connect(menu_, SIGNAL(activated(int)), SLOT(copySlot(int)));

	FolderViewItem *child = (FolderViewItem *)item->firstChild();
	while (child) {
		if (child->type() == FolderViewItem::Inbox)
			id = menu_->insertItem(SmallIcon("inbox"), child->name());
		else if (child->type() == FolderViewItem::Trash)
			id = menu_->insertItem(SmallIcon("trashcan_empty"), child->name());
		else
			id = menu_->insertItem(SmallIcon("folder"), child->name());

		folderMap.insert(id, child);

		if (child->firstChild()) {
			menu_->insertItem(SmallIcon("folder_open"), i18n("Subfolders of %1").arg(child->name()), rebuildFolderMenu(child, new QPopupMenu(), move));
		}

		child = (FolderViewItem *)child->nextSibling();
	}

	return menu_;
}


void FolderView::rebuildFolderMenus()
{
	folderMap.clear();
	moveMenu->clear();
	copyMenu->clear();

	(void)rebuildFolderMenu((FolderViewItem *)firstChild(), moveMenu, true);
	(void)rebuildFolderMenu((FolderViewItem *)firstChild(), copyMenu, false);
}


void FolderView::createFolder()
{
	EditFolder edit(this);
	edit.setCaption(i18n("New Folder"));

	if (edit.exec()) {
		QString name = edit.name();
		if (!name.isEmpty()) {
			QDir d(expandPath(current_folder->path() + "/" + name));
			if (d.exists()) {
				KMessageBox::sorry(this, i18n("Folder exists."), i18n("File Error"));
				return;
			}
			if (!d.mkdir(d.path())) {
				KMessageBox::sorry(this, i18n("Cannot create folder."), i18n("File Error"));
				return;
			}
			FolderViewItem *item = new FolderViewItem(current_folder, FolderViewItem::User);
			item->setName(name);
			rebuildFolderMenus();
		}
	}
}


void FolderView::renameFolder()
{
	EditFolder edit(this);
	edit.setCaption(i18n("Rename Folder"));
	edit.setName(current_folder->name());

	if (edit.exec()) {
		QString name = edit.name();
		if (!name.isEmpty()) {
			QDir d(expandPath(((FolderViewItem *)current_folder->parent())->path() + "/" + name));
			if (d.exists()) {
				KMessageBox::sorry(this, i18n("Folder exists."), i18n("File Error"));
				return;
			}
			if (!d.rename(expandPath(current_folder->path()), d.path())) {
				KMessageBox::sorry(this, i18n("Cannot rename folder."), i18n("File Error"));
				return;
			}
			current_folder->setName(name);
			rebuildFolderMenus();
		}
	}
}


void FolderView::removeFolder()
{
	if ((current_folder->files() > 0) || (currentItem()->childCount() > 0)) {
		KMessageBox::sorry(this, i18n("Folder is not empty."), i18n("File Error"));
		return;
	}
	QDir d;
	d.remove(expandPath(current_folder->path() + "/" + KMLOFAX_FAXDIR));
	if (!d.rmdir(expandPath(current_folder->path()))) {
		KMessageBox::sorry(this, i18n("Cannot remove folder."), i18n("File Error"));
		return;
	}

	delete current_folder;

	rebuildFolderMenus();
//	current_folder = 0;
}


void FolderView::rightButtonPressedSlot(QListViewItem *, const QPoint &point, int)
{
	if (current_folder)
		context_menu->popup(point);
}


void FolderView::currentChangedSlot(QListViewItem *item)
{
	if ((FolderViewItem *)item == current_folder)
		return;

	if (current_folder)
		current_folder->markCurrent(false);

	current_folder = (FolderViewItem *)item;

	if (current_folder) {
		current_folder->markCurrent(true);
		if (!dragging) {
			FolderViewItem::Type ftype = current_folder->type();
			context_menu->setItemEnabled(ID_FOLDER_NEW, (ftype == FolderViewItem::Root) || (ftype == FolderViewItem::User));
			context_menu->setItemEnabled(ID_FOLDER_RENAME, ftype == FolderViewItem::User);
			context_menu->setItemEnabled(ID_FOLDER_EMPTY, ftype != FolderViewItem::Root);
			context_menu->setItemEnabled(ID_FOLDER_REMOVE, ftype == FolderViewItem::User);
			emit hasCurrent(current_folder);
		}
	}
}


void FolderView::moveSlot(int id_)
{
	if (folderMap.contains(id_))
		emit move(folderMap[id_]);
}


void FolderView::copySlot(int id_)
{
	if (folderMap.contains(id_))
		emit copy(folderMap[id_]);
}


void FolderView::activatedSlot(int id_)
{
	switch (id_) {
		case ID_FOLDER_NEW:
			createFolder();
			break;
		case ID_FOLDER_RENAME:
			renameFolder();
			break;
		case ID_FOLDER_EMPTY:
			emit empty();
			break;
		case ID_FOLDER_REMOVE:
			removeFolder();
			break;
		default:
			break;
	}
}


void FolderView::contentsDragEnterEvent(QDragEnterEvent *e)
{
	if (!FaxDrag::canDecode(e)) {
		e->ignore();
		return;
	}
	dragging = true;
	old_item = currentItem();
	setFocus();
}


void FolderView::contentsDragMoveEvent(QDragMoveEvent *e)
{
	if (!FaxDrag::canDecode(e)) {
		e->ignore();
		return;
	}
	
	QListViewItem *i = itemAt(contentsToViewport(e->pos()));
	if (i && (((FolderViewItem *)i)->type() != FolderViewItem::Root)) {
		setCurrentItem(i);
		e->accept();
		switch (e->action()) {
			case QDropEvent::Copy:
			case QDropEvent::Move:
				e->acceptAction();
				break;
			default:
				break;
		}
	}
	else
		e->ignore();
}


void FolderView::contentsDragLeaveEvent(QDragLeaveEvent *)
{
	setCurrentItem(old_item);
	dragging = false;
	old_item = 0;
}


void FolderView::contentsDropEvent(QDropEvent *e)
{
	QListViewItem *i = itemAt(contentsToViewport(e->pos()));
	if (i && (((FolderViewItem *)i)->type() != FolderViewItem::Root)) {
		if (e->action() == QDropEvent::Copy)
			emit copy((FolderViewItem *)i);
		if (e->action() == QDropEvent::Move)
			emit move((FolderViewItem *)i);

		e->accept();
	}
	else
		e->ignore();
	setCurrentItem(old_item);
	dragging = false;
	old_item = 0;
}
