/*
 *
 * Copyright (C) 2002 George Staikos <staikos@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include "osdmanager.h"

#include "kdetvosdplugin.h"
#include "pluginfactory.h"
#include <kdebug.h>


OSDManager::OSDManager(PluginFactory* pf, QWidget* widget)
    : _osd(0L),
      _widget(widget),
      _pf(pf)
{
    scanPlugins();
}

OSDManager::~OSDManager()
{
    if (_osd) _osd->destroy();
}

void OSDManager::scanPlugins()
{
    if (!_widget) {
        if (_osd) _osd->destroy();
        _osd = 0L;
        kdWarning() << "OSDManager: No screenwidget set! Cannot scan OSD plugins!" << endl;
        return;
    }

    QPtrList<PluginDesc>& osdPlugins(_pf->osdPlugins());
    
    bool hasOSD = false;
    for (PluginDesc *plug = osdPlugins.first(); plug;
         plug = osdPlugins.next()) {
        kdDebug() << "OSDManager: Found a plugin: " << plug->name << endl;
        if (plug->enabled) {
            if (_osd) {
                if (_osd->pluginDescription() == plug) {
                    hasOSD = true;
                    break;
                } else {
                    _osd->destroy();
                    _osd = 0;
                }
            }
            _osd = _pf->getOSDPlugin(plug, _widget);
            if (_osd) {       // accept the first plugin we find
                connect(this, SIGNAL(colourKeyChanged(QColor)),
                        _osd, SLOT(colourKeyChanged(QColor)));
                hasOSD = true;
                break;
            }
        }
        else
            kdDebug() << "OSDManager: This plugin is disabled" << endl;
    }
    
    // if no enabled OSD found then delete old
    if (!hasOSD && _osd) {
        _osd->destroy();
        _osd = 0;
    }
}

void OSDManager::displayMisc(const QString& text)
{
    if (_osd)
        _osd->displayMisc(text);
}

void OSDManager::displayChannel(int channel, const QString& name)
{
    if (_osd)
        _osd->displayChannel(channel, name);
}

void OSDManager::displayMuted(bool muted)
{
    if (_osd)
        _osd->displayMuted(muted);
}

void OSDManager::displayVolume(int vol)
{
    if (_osd)
        _osd->displayVolume(vol);
}

void OSDManager::displayVolume(int left, int right)
{
    displayVolume((left + right) / 2);
}

void OSDManager::displayCC(const QString& text)
{
    if (_osd)
        _osd->displayCC(text);
}

QColor OSDManager::colourKey() const
{
    return _colourKey;
}

void OSDManager::setColourKey(QColor key)
{
    _colourKey = key;
    emit colourKeyChanged(key);
}

void OSDManager::setScreen(QWidget* screen)
{
    _widget = screen;
    scanPlugins();
}

#include "osdmanager.moc"

// vim: sw=4 ts=4
