/*
 * KHeiseReg
 *
 * A utility to search for articles within the Heise register.
 *
 * Copyright (C) 2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qlayout.h>
#include <qsize.h>
#include <qgroupbox.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qdir.h>

#include <kglobal.h>
#include <klocale.h>
#include <kapplication.h>
#include <kmessagebox.h>
#include <kstdguiitem.h>
#include <kiconloader.h>
#include <kstandarddirs.h>
#include <ktoolbar.h>
#include <kglobal.h>
#include <kdiroperator.h>

#include "global.h"
#include "regfile.h"
#include "appenddlg.h"



AppendFileDlg::AppendFileDlg(const QString &startDir, const QString &filter, QWidget *parent, const char *name, bool modal): KFileDialog(startDir, filter, parent, name, modal)
{
	config = kapp->config();

	goCt = new KAction(i18n("c't FTP directory"), "goct", 0, this, SLOT(slotGoCt()), actionCollection(), "go_ct");
	goIx = new KAction(i18n("iX FTP directory"), "goix", 0, this, SLOT(slotGoIx()), actionCollection(), "go_ix");

	toolBar()->insertSeparator();
	goCt->plug(toolBar());
	goIx->plug(toolBar());

	setOperationMode(Opening);
	setCaption(i18n("Open Heise Register File"));
	setMode(KFile::File);
}


AppendFileDlg::~AppendFileDlg()
{
}


KURL AppendFileDlg::getOpenURL(const QString &startDir, QWidget *parent)
{
	AppendFileDlg dlg(startDir, i18n("*.CTI *.cti *.FRM *.frm *.ZIP *.zip|Heise Register Files (*.cti, *.frm, *.zip)"), parent, "filedialog", true);

	dlg.setOperationMode(Opening);
	dlg.setCaption(i18n("Open Heise Register File"));
	dlg.setMode(KFile::File);
	dlg.ops->clearHistory();
	dlg.exec();

	return dlg.selectedURL();
}


void AppendFileDlg::slotGoCt()
{
	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_NETWORK));
	setURL(KURL(config->readEntry(ID_PREF_NET_CTURL, QString::fromLatin1(PREF_NET_CTURL))));
}


void AppendFileDlg::slotGoIx()
{
	config->setGroup(QString::fromLatin1(ID_PREF_GROUP_NETWORK));
	setURL(KURL(config->readEntry(ID_PREF_NET_IXURL, QString::fromLatin1(PREF_NET_IXURL))));
}



AppendDlg::AppendDlg(QWidget *parent, const char *name): KDialog(parent, name, true)
{
	QBoxLayout *vbox, *hbox, *svbox;
	QGridLayout *grid;
	QLabel *label;
	QGroupBox *gbox;
	KPushButton *bhelp, *bclose;
	QSize size;

	KStandardDirs::makeDir(expandPath(QString::fromLatin1(DOWNLOAD_DIR)));

	m_copyjob = 0;
	m_unzipproc = 0;

	vbox = new QVBoxLayout(this, marginHint());

	gbox = new QGroupBox(i18n("Database Status"), this);

	svbox = new QVBoxLayout(gbox, spacingHint());

	svbox->addSpacing(fontMetrics().height());

	grid = new QGridLayout(3, 5, spacingHint());
	svbox->addLayout(grid);

	label = new QLabel(i18n("First Edition"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 1, AlignVCenter | AlignRight);

	label = new QLabel(i18n("Last Edition"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 2, AlignVCenter | AlignRight);

	label = new QLabel(i18n("Editions"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 3, AlignVCenter | AlignRight);

	label = new QLabel(i18n("Articles"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 0, 4, AlignVCenter | AlignRight);

	label = new QLabel(i18n("c't Magazine"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 1, 0);

	ct_first = new QLabel(gbox);
	ct_first->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ct_first->setMinimumSize(ct_first->sizeHint());
	ct_first->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ct_first, 1, 1);

	ct_last = new QLabel(gbox);
	ct_last->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ct_last->setMinimumSize(ct_last->sizeHint());
	ct_last->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ct_last, 1, 2);

	ct_editions = new QLabel(gbox);
	ct_editions->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ct_editions->setMinimumSize(ct_editions->sizeHint());
	ct_editions->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ct_editions, 1, 3);

	ct_articles = new QLabel(gbox);
	ct_articles->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ct_articles->setMinimumSize(ct_articles->sizeHint());
	ct_articles->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ct_articles, 1, 4);

	label = new QLabel(i18n("iX Magazine"), gbox);
	label->setMinimumSize(label->sizeHint());
	grid->addWidget(label, 2, 0);

	ix_first = new QLabel(gbox);
	ix_first->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ix_first->setMinimumSize(ix_first->sizeHint());
	ix_first->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ix_first, 2, 1);

	ix_last = new QLabel(gbox);
	ix_last->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ix_last->setMinimumSize(ix_last->sizeHint());
	ix_last->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ix_last, 2, 2);

	ix_editions = new QLabel(gbox);
	ix_editions->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ix_editions->setMinimumSize(ix_editions->sizeHint());
	ix_editions->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ix_editions, 2, 3);

	ix_articles = new QLabel(gbox);
	ix_articles->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	ix_articles->setMinimumSize(ix_articles->sizeHint());
	ix_articles->setAlignment(AlignVCenter | AlignRight);
	grid->addWidget(ix_articles, 2, 4);

	svbox->activate();

	vbox->addWidget(gbox);

	vbox->addStretch();

	hbox = new QHBoxLayout(spacingHint());
	vbox->addLayout(hbox);

	bhelp = new KPushButton(KStdGuiItem::help(), this);
	bappend = new KPushButton(SmallIconSet(QString::fromLatin1("fileopen")), i18n("&Append"), this);
	bappend->setDefault(true);
	bclose = new KPushButton(KStdGuiItem::close(), this);
	bclose->setFixedWidth(bclose->sizeHint().width());

	size = bhelp->sizeHint().expandedTo(bappend->sizeHint());
	bhelp->setFixedSize(size);
	bappend->setFixedSize(size);
	bclose->setFixedHeight(size.height());

	hbox->addWidget(bhelp);
	hbox->addStretch();
	hbox->addWidget(bappend);
	hbox->addWidget(bclose);
	
	vbox->activate();

	setCaption(i18n("Append Register File"));

	connect(bhelp, SIGNAL(clicked()), this, SLOT(slotHelp()));
	connect(bappend, SIGNAL(clicked()), this, SLOT(slotAppend()));
	connect(bclose, SIGNAL(clicked()), this, SLOT(close()));

	scanEntries();
}


AppendDlg::~AppendDlg()
{
	if (m_copyjob)
		m_copyjob->kill();

	if (m_unzipproc)
		delete m_unzipproc;

	purgeDir();
}


bool AppendDlg::appendFile(const QString &filename)
{
	RegFile infile, regfile;
	RegEntry entry;
	int i;

	infile.setName(filename);
	if (!infile.open(IO_ReadOnly)) {
		KMessageBox::sorry(this, i18n("Can't open specified register file!"));
		return false;
	}

	if (!infile.scanEntries()) {
		infile.close();
		KMessageBox::sorry(this, i18n("Specified file is not a valid register file!\nError at line %1.").arg(infile.errorPos()));
		return false;
	}

	regfile.setName(expandPath(REGISTER_FILE));
	if (!regfile.open(IO_ReadWrite | IO_Append)) {
		infile.close();
		KMessageBox::sorry(this, i18n("Can't open internal data base file!"));
		return false;
	}
	regfile.scanEntries();

	if (regfile.containsEditions(REG_MAGAZINE_CT, infile.editions(REG_MAGAZINE_CT))) {
		regfile.close();
		infile.close();
		KMessageBox::sorry(this, i18n("Register file contains c't magazine editions\nwhich are already in the data base!"));
		return false;
	}

	if (regfile.containsEditions(REG_MAGAZINE_IX, infile.editions(REG_MAGAZINE_IX))) {
		regfile.close();
		infile.close();
		KMessageBox::sorry(this, i18n("Register file contains iX magazine editions\nwhich are already in the data base!"));
		return false;
	}

	while (!infile.atEnd()) {
		if (!infile.readEntry(&entry)) {
			KMessageBox::sorry(this, i18n("File read error!"));
			regfile.close();
			infile.close();
			return false;
		}

		for (i = 0; i < REG_LINE_COUNT; i++) {
			if (regfile.writeBlock(entry.line(i), (uint)strlen(entry.line(i))) == -1) {
				KMessageBox::sorry(this, i18n("File write error!"));
				regfile.close();
				infile.close();
				return false;
			}
		}
	}

	regfile.close();
	infile.close();

	return true;
}


void AppendDlg::slotHelp()
{
	kapp->invokeHelp(QString::fromLatin1("append-help"));
}


void AppendDlg::slotAppend()
{
	KURL url;

	url = AppendFileDlg::getOpenURL(QString::null, this);
	if (url.isEmpty())
		return;

	purgeDir();
	m_filename = expandPath(QString::fromLatin1(DOWNLOAD_DIR) + QString::fromLatin1("/") + url.fileName());

	m_copyjob = KIO::file_copy(url, KURL(QString::fromLatin1("file:") + m_filename), 0600, true, false, true);
	connect(m_copyjob, SIGNAL(result(KIO::Job *)), this, SLOT(slotCopyResult(KIO::Job *)));

	bappend->setEnabled(false);
}


void AppendDlg::slotCopyResult(KIO::Job *job)
{
	m_copyjob = 0;

	if (job->error()) {
		job->showErrorDialog(this);
		bappend->setEnabled(true);
		return;
	}

	if (m_filename.right(4).lower() == ".zip") {
		m_unzipproc = new KProcess;
		*m_unzipproc << "unzip";
		*m_unzipproc << "-joqqd" << expandPath(QString::fromLatin1(DOWNLOAD_DIR)) << m_filename;
		connect(m_unzipproc, SIGNAL(processExited(KProcess *)), this, SLOT(slotUnzipExited(KProcess *)));
		if (!m_unzipproc->start()) {
			KMessageBox::sorry(this, i18n("The unzip utility could not be started!"));
			purgeDir();
			bappend->setEnabled(true);
		}
	} else {
		appendFile(m_filename);
		purgeDir();
		scanEntries();
		bappend->setEnabled(true);
	}
}


void AppendDlg::slotUnzipExited(KProcess *proc)
{
	QFile::remove(m_filename);

	bappend->setEnabled(true);

	m_unzipproc = 0;

	if (!proc->normalExit()) {
		KMessageBox::sorry(this, i18n("The unzip process died!"));
		return;
	}
	if (proc->exitStatus()) {
		KMessageBox::sorry(this, i18n("The unzip process exited with status %1!").arg(proc->exitStatus()));
		return;
	}

	QDir d(expandPath(QString::fromLatin1(DOWNLOAD_DIR)));
	d.setNameFilter(QString::fromLatin1("*.cti *.CTI *.frm *.FRM"));
	const QFileInfoList *list = d.entryInfoList();
	if (list) {
		QFileInfoListIterator it(*list);
		QFileInfo *f;
		while ((f = it.current())) {
			if (!appendFile(f->filePath()))
				break;
			++it;
		}
	}

	purgeDir();
	scanEntries();
}


void AppendDlg::scanEntries()
{
	RegFile file(expandPath(REGISTER_FILE));

	ct_first->clear();
	ct_last->clear();
	ct_editions->clear();
	ct_articles->clear();
	ix_first->clear();
	ix_last->clear();
	ix_editions->clear();
	ix_articles->clear();

	if (file.open(IO_ReadOnly)) {
		QString s;

		if (file.scanEntries())
			file.close();
		else {
			file.close();
			if (KMessageBox::warningYesNo(this, i18n("The internal data base file is damaged!\nShall I remove this file (all entries will be lost)?")) == KMessageBox::Yes)
				QFile::remove(expandPath(REGISTER_FILE));
		}

		if (file.articles(REG_MAGAZINE_CT)) {
			s.sprintf("%d/%d", file.firstEdition(REG_MAGAZINE_CT) % 100, file.firstEdition(REG_MAGAZINE_CT) / 100);
			ct_first->setText(s);
			s.sprintf("%d/%d", file.lastEdition(REG_MAGAZINE_CT) % 100, file.lastEdition(REG_MAGAZINE_CT) / 100);
			ct_last->setText(s);
			ct_editions->setText(KGlobal::locale()->formatNumber((double)file.editionCount(REG_MAGAZINE_CT), 0));
			ct_editions->setMinimumSize(ct_editions->sizeHint());
			ct_articles->setText(KGlobal::locale()->formatNumber((double)file.articles(REG_MAGAZINE_CT), 0));
			ct_articles->setMinimumSize(ct_articles->sizeHint());
		}
		if (file.articles(REG_MAGAZINE_IX)) {
			s.sprintf("%d/%d", file.firstEdition(REG_MAGAZINE_IX) % 100, file.firstEdition(REG_MAGAZINE_IX) / 100);
			ix_first->setText(s);
			s.sprintf("%d/%d", file.lastEdition(REG_MAGAZINE_IX) % 100, file.lastEdition(REG_MAGAZINE_IX) / 100);
			ix_last->setText(s);
			ix_editions->setText(KGlobal::locale()->formatNumber((double)file.editionCount(REG_MAGAZINE_IX), 0));
			ix_editions->setMinimumSize(ix_editions->sizeHint());
			ix_articles->setText(KGlobal::locale()->formatNumber((double)file.articles(REG_MAGAZINE_IX), 0));
			ix_articles->setMinimumSize(ix_articles->sizeHint());
		}
	}
}


void AppendDlg::purgeDir()
{
	QDir d(expandPath(QString::fromLatin1(DOWNLOAD_DIR)));
	d.setFilter(QDir::Files | QDir::Hidden | QDir::NoSymLinks);
	const QFileInfoList *list = d.entryInfoList();
	if (list) {
		QFileInfoListIterator it(*list);
		QFileInfo *f;
		while ((f = it.current())) {
			QFile::remove(f->filePath());
			++it;
		}
	}
}
