// Qt includes
#include <qlayout.h>
#include <qwidget.h>
#include <qsplitter.h>

// KDE includes
#include <kmainwindow.h>
#include <kapp.h>
#include <kconfig.h>
#include <kstdaction.h>
#include <kaction.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kedittoolbar.h>
#include <kfiledialog.h>

// KMtraceViewer includes
#include "KMtraceViewerMainWindow.hh"
#include "KMtraceViewerMainWindow.moc"
#include "KMtraceModel.hh"
#include "KMtraceStatisticsView.hh"
#include "KMtraceLeaksView.hh"

KMtraceViewerMainWindow::KMtraceViewerMainWindow( const char *ktreefile )
  : KMainWindow( 0L, "kmtraceviewer main window" )
{
   QWidget *mainWidget = new QWidget( this );
   QGridLayout *layout = new QGridLayout( mainWidget, 2, 3 );

   // set to the empty model (until we read a valid ktreefile)
   m_model = 0;

   // second part is the statistics section (all a big label, this view has
   // a shared common model, the kout model)
   m_statsView = new KMtraceStatisticsView( mainWidget, m_model );
   layout->addWidget( m_statsView, 2, 0 );

   // allow the user to choose how much space he needs for the leaks and
   // how much for the source code
   m_splitter = new QSplitter( QSplitter::Vertical, mainWidget );
   layout->addWidget( m_splitter, 3, 0 );

   // the third part is the leak view (a list view)
   m_leaksView = new KMtraceLeaksView( m_splitter, m_model );
   
   // add an edit window (for showing the point of the leak being produced)
   m_sourceCodeView = new KMtraceSourceCodeView( m_splitter );
   	
   // status bar showing position and source file being shown
   m_statusBar = new KStatusBar( mainWidget );     
   layout->addWidget( m_statusBar, 4, 0 );

   /// the file menu
   KStdAction::open( this, SLOT( slotOpenFile( ) ), actionCollection( ) );
   KStdAction::openRecent( this, SLOT( slotOpenRecentFile( const KURL& ) ),
                           actionCollection( ) );
   KStdAction::quit( kapp, SLOT( closeAllWindows( ) ), actionCollection( ) );

   // the leaks menu
   (void)new KAction( i18n( "&Expand all" ), 0, m_leaksView,
                      SLOT( slotExpandAll( ) ), actionCollection( ),
                      "expand_all" );
   (void)new KAction( i18n( "&Collapse all" ), 0, m_leaksView,
                      SLOT( slotCollapseAll( ) ), actionCollection( ),
                      "collapse_all" );
   m_showSuppressListDialog = new KToggleAction( i18n( "&Suppressed leaks" ),
                        0, this, SLOT( slotSuppressListDialog( ) ),
                        actionCollection( ), "suppress" );
   m_showSuppressedAction = new KToggleAction( i18n( "S&how suppressed leaks" ), 0,
                            this, SLOT( slotShowSuppressed( ) ),
                            actionCollection( ),
                            "show_suppressed" );

   // find menu
   KStdAction::find( m_leaksView, SLOT( slotFind( ) ), actionCollection( ) );
   
   // prefereces menu
//   KStdAction::preferences( kapp, SLOT( slotShowPreferences( ) ), actionCollection( ) );
//   KStdAction::configureToolbars( this, SLOT( slotEditToolbars( ) ), actionCollection( ) );

   // create user interface (with rc file as factory, keep resources for edit dialog!)
   createGUI( "kmtraceviewer.rc", false );
   
   // all happens in the main window
   setCentralWidget( mainWidget );
   
   // we want to get informed if the leak view show a different leak
   // position (module, line)
   connect( m_leaksView, SIGNAL( moduleChanged( QString, int ) ),
            this, SLOT( slotModuleChanged( QString, int ) ) );
   connect( m_leaksView, SIGNAL( moduleUnselect( ) ),
            this, SLOT( slotModuleUnselect( ) ) );	    
   connect( m_leaksView, SIGNAL( moduleChanged( QString, int ) ),
            m_sourceCodeView, SLOT( slotModuleChanged( QString, int ) ) );
   connect( m_leaksView, SIGNAL( moduleUnselect( ) ),
            m_sourceCodeView, SLOT( slotModuleUnselect( ) ) );	    

   m_suppressListDialog = 0;

   // open the model (the kmtrace output)
   setModelFromFilename( ktreefile );

   // load configuration
   KConfig *config = kapp->config( );
   KRecentFilesAction *recent = (KRecentFilesAction*)
      actionCollection( )->action( KStdAction::stdName( KStdAction::OpenRecent) );
   recent->loadEntries( config );

   // set some default sizes
   QValueList<int> valueList;
   valueList.append( 200 );
   m_splitter->setSizes( valueList );
   setGeometry( 100, 100, 640, 400 );

   restoreSettings( kapp->config( ) );
}

KMtraceViewerMainWindow::~KMtraceViewerMainWindow( )
{
   saveSettings( kapp->config( ) );
}

void KMtraceViewerMainWindow::slotModuleChanged( QString module, int line )
{
   QString message( i18n( "Leak in %1, line %2" ).arg( module ).arg( line ) );
   m_statusBar->message( message );
}

void KMtraceViewerMainWindow::slotModuleUnselect( )
{
   m_statusBar->message( i18n( "No source code available!" ) );
}

void KMtraceViewerMainWindow::slotEditToolbars( )
{
   KEditToolbar dlg( actionCollection( ), "kmtraceviewer.rc" );
   connect( &dlg, SIGNAL( newToolbarConfig( ) ),
            SLOT(slotUpdateToolbars( ) ) );

   dlg.exec();
}

void KMtraceViewerMainWindow::slotUpdateToolbars( )
{
   createGUI(" kmtraceviewer.rc" );
}

void KMtraceViewerMainWindow::slotShowSuppressed( )
{
   m_leaksView->setShowSuppressed( m_showSuppressedAction->isChecked( ) );
}

void KMtraceViewerMainWindow::slotSuppressListDialog( )
{
   if( m_showSuppressListDialog->isChecked( ) ) {
      if( m_suppressListDialog == 0 ) {
         m_suppressListDialog = new KMtraceSuppressListDialog( m_model, this,
                                  "suppress_list_dlg", false );
         connect( m_suppressListDialog, SIGNAL( done( ) ),
                  this, SLOT( slotSuppressListDialogDone( ) ) );
      }
      m_suppressListDialog->show( );
      m_suppressListDialog->result( );
   } else if( m_suppressListDialog ) {
      m_suppressListDialog->hide( );
      this->setFocus( );
   }
}

void KMtraceViewerMainWindow::slotSuppressListDialogDone( )
{
   if( m_suppressListDialog == 0 ) {
      return;
   }
   
   m_suppressListDialog->hide( );
   this->setFocus( );
   m_showSuppressListDialog->setChecked( false );
}

void KMtraceViewerMainWindow::slotOpenFile( )
{
   QString filename;

   filename = KFileDialog::getOpenFileName( 0, "*.kout", this,
              i18n( "Select a kmtrace output file" ) );
   if( !filename.isEmpty( ) ) {
      setModelFromFilename( filename );
   }
}

void KMtraceViewerMainWindow::slotOpenRecentFile( const KURL& url )
{
   QString filename = url.path( );
   setModelFromFilename( filename );
}

void KMtraceViewerMainWindow::setModelFromFilename( const char* filename )
{
   KMtraceModel *model = 0;

   if( filename ) {
      model = new KMtraceModel( filename );
   }
   setModel( model );

   // inform as a recently opened file
   KURL url( filename );
   KRecentFilesAction *recent = (KRecentFilesAction*)
      actionCollection( )->action( KStdAction::stdName( KStdAction::OpenRecent) );
   recent->addURL( url );

   if( url.fileName( ).isEmpty( ) ) {
      setCaption( filename );
   } else {
      setCaption( url.fileName( ) );
   }
}

void KMtraceViewerMainWindow::setModel( KMtraceModel *model )
{
   if( m_model ) {
      delete m_model;
   }
   m_model = model;
   m_statsView->setModel( m_model );
   m_leaksView->setModel( m_model );

   if( m_model ) {
      // if somebody changes suppression we must inform the statistics view
      connect( m_model, SIGNAL( changed( ) ),
               m_statsView, SLOT( slotModelChanged( ) ) );
      m_model->applySuppressions( );
   }
}
      
bool KMtraceViewerMainWindow::queryExit( )
{
   KConfig *config = kapp->config( );
   KRecentFilesAction *recent = (KRecentFilesAction*)
      actionCollection( )->action( KStdAction::stdName( KStdAction::OpenRecent) );
   recent->saveEntries( config );
   delete m_model;
   return true;
}

void KMtraceViewerMainWindow::saveSettings( KConfig *config )
{
   int count = 0;
   config->setGroup( "Geometry" );
   saveWindowSize( config );
   m_leaksView->saveLayout( config, "LeakViewGeometry" );
   config->setGroup( "Splitter" );
   QValueList<int> list = m_splitter->sizes( );
   QValueList<int>::Iterator it = list.begin( );
   while( it != list.end( ) ) {
      QString result;
      QTextOStream( &result ) << "rulerOffset" << count;
      config->writeEntry( result, *it );
      ++count;
      ++it;
   }
   config->setGroup( "MenuSettings" );
   config->writeEntry( "ShowSuppressed", m_showSuppressedAction->isChecked( ) );
   config->writeEntry( "ShowSuppressedListDialog",
                       m_showSuppressListDialog->isChecked( ) );
}

void KMtraceViewerMainWindow::restoreSettings( KConfig *config )
{
   m_leaksView->restoreLayout( config, "LeakViewGeometry" );
   config->setGroup( "Geometry" );
   restoreWindowSize( config );
   config->setGroup( "Splitter" );
   QValueList<int> valueList;
   int count = 0;
   while( true ) {
      QString result;
      QTextOStream( &result ) << "rulerOffset" << count;
      int rulerOffset = config->readNumEntry( result, -1 );
      if( rulerOffset == -1 ) {
         break;
      }
      valueList.append( rulerOffset );
      count++;
   }
   m_splitter->setSizes( valueList );
   config->setGroup( "MenuSettings" );
   bool showSuppressed = config->readBoolEntry( "ShowSuppressed" );
   m_showSuppressedAction->setChecked( showSuppressed );
   m_leaksView->setShowSuppressed( showSuppressed );
   bool suppressListDialogActive = config->readBoolEntry( "ShowSuppressedListDialog" );
   m_showSuppressListDialog->setChecked( suppressListDialogActive );
   if( suppressListDialogActive ) {
      slotSuppressListDialog( );
   }
}
