/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef CATEGORY_H
#define CATEGORY_H

// WorKflow includes
#include "xmldescription.h"

// forward declarations
namespace WorKflow
{
  class LibraryDescription;
}

namespace WorKflow
{
  /**
   * @short Represents a command category.
   *
   * @section overview Overview
   *
   * This class stores information about a command category. All information
   * about a category is stored in the command library's XML file.
   *
   * Since Category inherits from XMLDescription, you can get the category's
   * id(), the name() and the description() using the methods from the base
   * class. Additionally, a category can have an assigned icon. The icon's
   * name is returned by iconName(); you can set it with setIconName().
   *
   * To get all commands of a certain category, use CommandManager::commands().
   *
   * @section xml The XML file
   *
   * Usually, you should not need to deal with this class at all, because
   * the XML file contains all information that is neccessary for categories
   * to work. The category is automatically registered with the CommandManager
   * class upon creation.
   *
   * Example:
   * @code
   * <category id="org.kde.public.ListManipulation" >
   *     <name>List Manipulation</name>
   *     <description>Commands that manipulate lists of values</description>
   * </category>
   * @endcode
   *
   * Each command must be at least in one category, but can optionally be in
   * more than one categories. Commands are assigned to categories through
   * the library's XML file as well.
   *
   * Example:
   * @code
   * <command id="org.kde.Standard.ShowMessage" >
   *     <name>Show Message</name>
   *     <description>Pops up a message to the user.</description>
   *     <categories>
   *         <category id="org.kde.public.Debug" />
   *         <category id="org.kde.public.Misc" />
   *     </categories>
   *     ...
   * </command>
   * @endcode
   *
   * @section xmlref XML Tag Reference
   *
   * The following XML tags and attributes are recognized for categories:
   *
   * - @c category The category tag, required. The required @c id attribute defines
   *   the category's ID. The id is accessible through the id() method.
   *   - @c name The category's name, in english. This will be translated to the
   *     user's current language. Use name() to get this translated text. Required.
   *   - @c description A short help text, in english. This will be translated to
   *     the user's current languate. Use description() to get this text. Required.
   *   - @c icon The icon to use. The @c name attribute contains the name of the
   *     icon. Optional.
   *
   * @sa CommandManager
   *
   * @author Thomas Kadauke
   */
  class Category : public XMLDescription
  {
  public:
    /**
     * Constructor. Registers the command with the CommandManager class.
     * @param parent The library's description object to which the category
     *   belongs.
     */
    Category(LibraryDescription* parent);

    /**
     * Destructor. Unregisters the category.
     */
    ~Category();

    /**
     * Returns the category's icon name.
     * @return The category's icon name.
     */
    QString iconName() const;

    /**
     * Sets the category's icon name.
     * @param iconName The new icon name.
     */
    void setIconName(const QString& iconName);

    /**
     * Reimplemented from XMLDescription. Reads the category's information from
     * the library's XML file.
     * @param e The category's XML tag node.
     */
    virtual void readXML(const QDomElement& e);

    /**
     * Reimplemented from XMLDescription. Writes the category's information to
     * the library's XML file.
     * @param doc The XML document object.
     * @param e The category's parent tag node (i.e. the library tag).
     */
    virtual void writeXML(QDomDocument& doc, QDomElement& e);

  private:
    class Private;
    Private* d;
  };
}

#endif
