/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>

// Qt includes
#include <qdom.h>

// WorKflow includes
#include "parameter.h"
#include "command.h"
#include "typemanager.h"
#include "commanddescription.h"
#include "commandmanager.h"
#include "parameterdescription.h"
#include "datatype.h"

using namespace WorKflow;

Parameter::Parameter(Command* command, ParameterDescription* desc)
  : Slot(command, desc), m_kind(Unset)
{
  command->addParameter(this);

  setTypeIds(desc->typeIds());
  m_source = desc->source();
  m_value = Value(desc->typeIds().first());

  if (m_value.type() && m_value.type()->isEnum())
    m_value.addString(m_value.type()->enumKeys().first());

  m_optional = desc->isOptional();
}

Parameter::~Parameter()
{

}

Parameter::Kind Parameter::kind()
{
  return m_kind;
}

bool Parameter::isSet()
{
  return m_kind != Unset;
}

void Parameter::clear()
{
  m_kind = Unset;
  m_value.clear();
  emit changed(m_value);
}

void Parameter::reset()
{
  if (m_kind == Piped)
    m_value.clear();
}

void Parameter::setStaticValue(const Value& value)
{
  if (value != m_value) {
    m_kind = Static;
    setValue(value);
    emit changed(value);
  }
}

Value Parameter::value()
{
  return m_value;
}

void Parameter::setValue(const Value& value)
{
  m_value = value;
}

void Parameter::readXML(const QDomElement& e)
{
  QString tid = e.attribute("datatype");
  if (!tid.isNull()) {
    setType(tid);
  }

  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "value") {
        m_value.readXML(e);
        emit changed(m_value);
      }
    }

    n = n.nextSibling();
  }
}

void Parameter::writeXML(QDomDocument& doc, QDomElement& e)
{
  QDomElement paramTag = doc.createElement("param");
  paramTag.setAttribute("id", id());
  paramTag.setAttribute("datatype", typeId());

  if (m_source != ConnectionSource)
    m_value.writeXML(doc, paramTag);

  e.appendChild(paramTag);
}

Parameter::Source Parameter::source()
{
  return m_source;
}

bool Parameter::isOptional()
{
  return m_optional;
}

bool Parameter::assignType(Datatype* type)
{
  QStringList typeids = typeIds();
  for (QStringList::ConstIterator i = typeids.begin(); i != typeids.end(); ++i) {
    Datatype* t = TypeManager::self()->find(*i);
    if (type->inherits(t)) {
      setType(type->id());
      return true;
    } else if (type->isConvertibleTo(t)) {
      setType(t->id());
      return true;
    }
  }

  return false;
}

#include "parameter.moc"
