/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef PARAMETER_H
#define PARAMETER_H

// Qt includes
#include <qstring.h>

// WorKflow includes
#include "slot.h"

// forward declarations
class QDomElement;
class QDomDocument;

namespace WorKflow
{
  class Command;
  class Datatype;
  class Result;
  class ParameterDescription;
}

namespace WorKflow
{
  /**
   * @short Represents a Command's parameter slot.
   *
   * This class represents a parameter slot for a command instance. A command has
   *  one or more of these. There are two ways to assign a value to a parameter:
   *
   * - Set the parameter value statically, using setStaticValue(),
   * - Connect the parameter with a Result instance from another command, such
   *   that data flows automatically, using the connect() method.
   *
   * The kind() method returns how the value is assigned to the parameter,
   * according to the above list. The reset() method is used to erase any
   * dynamically assigned value. The value() method returns the parameter's
   * value, independent of the method's Kind.
   *
   * See the Slot class for more methods.
   */
  class Parameter : public Slot
  {
    Q_OBJECT

  public:
    /**
     * This enum describes how the parameter gets its value assigned.
     */
    enum Kind
    {
      Unset,      ///< Nothing is assigned to the parameter.
      Static,     ///< The value is assigned statically.
      Piped,      ///< The value is assigned dynamically, i.e. the Parameter
                  ///< is connected to a Result.
    };

    enum Source
    {
      GuiSource,
      ConnectionSource,
      AllSources,
    };

    /**
     * Constructor.
     * @param command The command to which the parameter is associated.
     * @param desc The parameter's description
     */
    Parameter(Command* command, ParameterDescription* desc);

    /**
     * Destructor.
     */
    ~Parameter();

    /**
     * Returns the parameter's kind.
     * @return The parameter's kind.
     * @sa Kind
     */
    Kind kind();

    Source source();

    /**
     * Returns whether a value or a Result is assigned to the parameter.
     * @return Whether the parameter is set.
     */
    bool isSet();

    /**
     * Clears the parameter's contents. Afterwards, the parameter is Unset.
     */
    void clear();

    /**
     * Resets the parameter slot after an execution of a workflow such that any
     * dynamically assigned value is erased.
     */
    void reset();

    /**
     * Returns the parameter's statically or dynamically assigned value.
     * @return The parameter's value.
     */
    Value value();

    bool isOptional();
    bool assignType(Datatype* type);

    void readXML(const QDomElement& e);
    void writeXML(QDomDocument& doc, QDomElement& e);

  public slots:
    /**
     * Sets the parameter's Kind to Static and assignes a static value.
     * @param value The static value.
     */
    void setStaticValue(const Value& value);

  signals:
    void changed(const Value& value);

  private:
    friend class Result;

    void setValue(const Value& value);

    Kind m_kind;
    Source m_source;
    Value m_value;
    bool m_optional;
  };
}

#endif
