/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <klocale.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <klistview.h>

// Qt includes
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>

// WorKflow includes
#include "parameterdescription.h"
#include "commanddescription.h"
#include "parameter.h"
#include "typemanager.h"
#include "datatype.h"

// designer includes
#include "parameterdialog.h"
#include "dialoghelper.h"
#include "selecttypedialog.h"
#include "typelistitem.h"

using namespace WorKflow;

bool ParameterDialog::editParameter(QWidget* parent, CommandDescription* cmd, ParameterDescription* desc)
{
  ParameterDialog dialog(parent, cmd, desc);
  return dialog.exec();
}

ParameterDescription* ParameterDialog::newParameter(QWidget* parent, CommandDescription* cmd)
{
  ParameterDialog dialog(parent, cmd, new ParameterDescription(cmd));
  if (dialog.exec())
    return dialog.m_description;
  else
    return 0;
}

// initialization
void ParameterDialog::init()
{
  m_idEdit->setText(m_description->id());
  m_nameEdit->setText(m_description->name());
  m_descriptionEdit->setText(m_description->description());
  m_optionalCheckBox->setChecked(m_description->isOptional());

  Parameter::Source source = m_description->source();

  switch (source) {
    case Parameter::GuiSource:
      m_sourceComboBox->setCurrentItem(0);
      break;
    case Parameter::ConnectionSource:
      m_sourceComboBox->setCurrentItem(1);
      break;
    case Parameter::AllSources:
      m_sourceComboBox->setCurrentItem(2);
      break;
  }

  QStringList types = m_description->typeIds();
  for (QStringList::ConstIterator i = types.begin(); i != types.end(); ++i) {
    Datatype* type = TypeManager::self()->find(*i);
    if (type) {
      new TypeListItem(m_typesListView, type);
    }
  }
}

// save description
void ParameterDialog::accept()
{
  QString id = m_idEdit->text();
  if (id.isEmpty()) {
    KMessageBox::error(this, i18n("Please enter an ID"), i18n("ID missing"));
    m_idEdit->setFocus();
    return;
  } else {
    ParameterDescription* param = m_command->parameterDescription(id);
    if (param && param != m_description) {
      KMessageBox::error(this, i18n("The ID you entered is already taken"), i18n("ID not unique"));
      m_idEdit->setFocus();
      return;
    }
  }

  if (m_nameEdit->text().isEmpty()) {
    KMessageBox::error(this, i18n("Please enter a name"), i18n("Name missing"));
    m_nameEdit->setFocus();
    return;
  }

  m_description->setId(id);
  m_description->setName(m_nameEdit->text());
  m_description->setDescription(m_descriptionEdit->text());
  m_description->setOptional(m_optionalCheckBox->isChecked());

  switch (m_sourceComboBox->currentItem()) {
    case 0:
      m_description->setSource(Parameter::GuiSource);
      break;
    case 1:
      m_description->setSource(Parameter::ConnectionSource);
      break;
    case 2:
      m_description->setSource(Parameter::AllSources);
      break;
  }

  m_description->clearTypeIds();
  QListViewItemIterator it(m_typesListView);
  while (it.current()) {
    TypeListItem* item = static_cast<TypeListItem*>(it.current());
    m_description->addTypeId(item->type()->id());
    ++it;
  }

  QDialog::accept();
}

ParameterDialog::ParameterDialog(QWidget* parent, CommandDescription* cmd, ParameterDescription* desc)
  : ParameterDialogBase(parent), m_command(cmd), m_description(desc)
{
  init();
}

ParameterDialog::~ParameterDialog()
{
}

// type list
void ParameterDialog::addType()
{
  Datatype* type = SelectTypeDialog::select(this);
  if (type) {
    TypeListItem* item = new TypeListItem(m_typesListView, type);
    m_typesListView->setSelected(item, true);
  }
}

void ParameterDialog::removeType()
{
  DialogHelper::removeItem(m_typesListView);
}

void ParameterDialog::slotTypeSelectionChanged()
{
  QListViewItem* item = m_typesListView->selectedItem();
  m_removeTypeButton->setEnabled(item != 0);
  m_typeUpButton->setEnabled(item != 0 && item->itemAbove() != 0);
  m_typeDownButton->setEnabled(item != 0 && item->itemBelow() != 0);
}

void ParameterDialog::typeDown()
{
  DialogHelper::itemDown(m_typesListView);
}

void ParameterDialog::typeUp()
{
  DialogHelper::itemUp(m_typesListView);
}

#include "parameterdialog.moc"

