/*
    This file is part of KOrganizer.
    Copyright (c) 2001,2002 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

#include <qtooltip.h>
#include <qframe.h>
#include <qpixmap.h>
#include <qlayout.h>
#include <qwidgetstack.h>

#include <kiconloader.h>
#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>

#include <libkdepim/categoryselectdialog.h>
#include <libkcal/calendarlocal.h>

#include "koprefs.h"
#include "kogroupware.h"
#include "koeventtimewidget.h"

#include "koeventeditor.h"
#include "koeventeditor.moc"



void DateTimeMediator::slotDateTimesChanged( QDateTime fromDt, QDateTime toDt )
{
    if( fromDt == mCurrStartDateTime &&
        toDt == mCurrEndDateTime )
        return; // no change

    mCurrStartDateTime = fromDt;
    mCurrEndDateTime = toDt;

    // Compute the new date/time string
    KLocale* l = KGlobal::locale();

    QString from,to;
    if (mAllDay) {
        from = l->formatDate(mCurrStartDateTime.date());
        to = l->formatDate(mCurrEndDateTime.date());
    } else {
        from = l->formatDateTime(mCurrStartDateTime);
        to = l->formatDateTime(mCurrEndDateTime);
    }

    QString str = i18n("From: %1   To: %2   %3").arg(from).arg(to)
                  .arg(mDuration);

    emit dateTimesChanged( mCurrStartDateTime, mCurrEndDateTime,
                           str, false );
}


void DateTimeMediator::slotAllDayChanged( bool allDay )
{
    if( allDay == mAllDay )
        return; // no change

    mAllDay = allDay;

    emit allDayChanged( mAllDay, false );
}


void DateTimeMediator::slotRecursChanged( bool recurs )
{
    if( recurs == mRecurs )
        return; // no change

    mRecurs = recurs;

    emit recursChanged( mRecurs, false );
}



KOEventEditor::KOEventEditor( Calendar *calendar, QWidget *parent ) :
  KOIncidenceEditor( i18n("Edit Event"), calendar, parent )
{
  mEvent = 0;
}

KOEventEditor::~KOEventEditor()
{
  emit dialogClose( mEvent );
}

void KOEventEditor::init()
{
  setupGeneral();
  setupAttendeesTab( i18n( "Attendees" ), true );

  // Groupware connections
  if( KOPrefs::instance()->mGroupwareCommunication ) {
    mDateTimeMediator = new DateTimeMediator( this );
    // General page connections
    connect( mGeneral->eventTimeWidget(),
	     SIGNAL( dateTimesChanged( QDateTime, QDateTime ) ),
	     mDateTimeMediator,
	     SLOT( slotDateTimesChanged( QDateTime, QDateTime ) ) );
    connect( mDateTimeMediator,
	     SIGNAL( dateTimesChanged( QDateTime, QDateTime,
				       const QString&, bool ) ),
	     mGeneral->eventTimeWidget(),
	     SLOT( slotDateTimesChanged( QDateTime, QDateTime,
					 const QString&, bool ) ) );
    connect( mGeneral->eventTimeWidget(),
	     SIGNAL( allDayChanged( bool ) ),
	     mDateTimeMediator,
	     SLOT( slotAllDayChanged( bool ) ) );
    connect( mDateTimeMediator,
	     SIGNAL( allDayChanged( bool, bool ) ),
	     mGeneral->eventTimeWidget(),
	     SLOT( setAllDay( bool, bool ) ) );
    connect( mGeneral->eventTimeWidget(),
	     SIGNAL( recursChanged( bool ) ),
	     mDateTimeMediator,
	     SLOT( slotRecursChanged( bool ) ) );
    connect( mDateTimeMediator,
	     SIGNAL( recursChanged( bool, bool ) ),
	     mGeneral->eventTimeWidget(),
	     SLOT( setRecurs( bool, bool ) ) );
    connect( mDetails->eventTimeWidget(),
	     SIGNAL( dateTimesChanged( QDateTime, QDateTime ) ),
	     mDateTimeMediator,
	     SLOT( slotDateTimesChanged( QDateTime, QDateTime ) ) );
    connect( mDateTimeMediator,
	     SIGNAL( dateTimesChanged( QDateTime, QDateTime,
				       const QString&, bool ) ),
	     mDetails->eventTimeWidget(),
	     SLOT( slotDateTimesChanged( QDateTime, QDateTime,
					 const QString&, bool ) ) );
    connect( mDetails->eventTimeWidget(),
	     SIGNAL( allDayChanged( bool ) ),
	     mDateTimeMediator,
	     SLOT( slotAllDayChanged( bool ) ) );
    connect( mDateTimeMediator,
	     SIGNAL( allDayChanged( bool, bool ) ),
	     mDetails->eventTimeWidget(),
	     SLOT( setAllDay( bool, bool ) ) );
    connect( mDetails->eventTimeWidget(),
	     SIGNAL( recursChanged( bool ) ),
	     mDateTimeMediator,
	     SLOT( slotRecursChanged( bool ) ) );
    connect( mDateTimeMediator,
           SIGNAL( recursChanged( bool, bool ) ),
	     mDetails->eventTimeWidget(),
	     SLOT( setRecurs( bool, bool ) ) );
    connect( mDateTimeMediator,
	     SIGNAL( dateTimesChanged( QDateTime, QDateTime, const QString&, bool ) ),
	     mDetails,
	     SLOT( slotUpdateGanttView( QDateTime, QDateTime ) ) );
  }

#ifndef KORG_NORECURRENCE
  setupRecurrence();

  // Propagate date time settings to recurrence tab
  if( KOPrefs::instance()->mGroupwareCommunication ) {
    connect( mDateTimeMediator,
	     SIGNAL( dateTimesChanged( QDateTime, QDateTime,
				       const QString&, bool ) ),
	     mRecurrence,
	     SLOT( setDateTimes( QDateTime, QDateTime ) ) );

    // Enable/Disable recurrence tab
    connect(mDateTimeMediator,SIGNAL(recursChanged(bool,bool)),
	    SLOT(enableRecurrence(bool)));
  } else {
    connect(mGeneral,SIGNAL(dateTimesChanged(QDateTime,QDateTime)),
	    mRecurrence,SLOT(setDateTimes(QDateTime,QDateTime)));
    connect(mGeneral,SIGNAL(dateTimeStrChanged(const QString &)),
	    mRecurrence,SLOT(setDateTimeStr(const QString &)));

    // Enable/Disable recurrence tab
    connect(mGeneral,SIGNAL(recursChanged(bool)),
	    SLOT(enableRecurrence(bool)));
  }
#endif

  // Category dialog
  connect(mGeneral,SIGNAL(openCategoryDialog()),mCategoryDialog,SLOT(exec()));
  connect(mCategoryDialog,SIGNAL(categoriesSelected(const QString &)),
          mGeneral,SLOT(setCategories(const QString &)));
}

void KOEventEditor::reload()
{
  if ( mEvent ) readEvent( mEvent );
}

void KOEventEditor::setupGeneral()
{
  mGeneral = new KOEditorGeneralEvent( this );

  if( KOPrefs::instance()->mCompactDialogs ) {
    QFrame *topFrame = addPage(i18n("General"));

    QBoxLayout *topLayout = new QVBoxLayout(topFrame);
    topLayout->setMargin(marginHint());
    topLayout->setSpacing(spacingHint());

    mGeneral->initHeader(topFrame,topLayout);
    mGeneral->initTime(topFrame,topLayout);
//    QBoxLayout *alarmLineLayout = new QHBoxLayout(topLayout);
    mGeneral->initAlarm(topFrame,topLayout);
    mGeneral->enableAlarm( false );
    mGeneral->initCategories( topFrame, topLayout );

    topLayout->addStretch( 1 );

    QFrame *topFrame2 = addPage(i18n("Details"));

    QBoxLayout *topLayout2 = new QVBoxLayout(topFrame2);
    topLayout2->setMargin(marginHint());
    topLayout2->setSpacing(spacingHint());

    mGeneral->initClass(topFrame2,topLayout2);
#if 0
    mGeneral->initSecrecy( topFrame2, topLayout2 );
#endif
    mGeneral->initDescription(topFrame2,topLayout2);
  } else {
    QFrame *topFrame = addPage(i18n("General"));

    QBoxLayout *topLayout = new QVBoxLayout(topFrame);
    topLayout->setMargin(marginHint());
    topLayout->setSpacing(spacingHint());

    mGeneral->initHeader(topFrame,topLayout);
    mGeneral->initTime(topFrame,topLayout);
    QBoxLayout *alarmLineLayout = new QHBoxLayout(topLayout);
    mGeneral->initAlarm(topFrame,alarmLineLayout);
    mGeneral->initClass(topFrame,alarmLineLayout);
    mGeneral->initDescription(topFrame,topLayout);
    QBoxLayout *detailsLayout = new QHBoxLayout(topLayout);
    mGeneral->initCategories( topFrame, detailsLayout );
#if 0
    mGeneral->initSecrecy( topFrame, detailsLayout );
#endif
  }

  mGeneral->finishSetup();
}

void KOEventEditor::setupRecurrence()
{
  QFrame *topFrame = addPage(i18n("Recurrence"));

  QBoxLayout *topLayout = new QVBoxLayout(topFrame);
  topLayout->setMargin(marginHint());

  mRecurrenceStack = new QWidgetStack(topFrame);
  topLayout->addWidget(mRecurrenceStack);

  mRecurrence = new KOEditorRecurrence(spacingHint(),mRecurrenceStack);
  mRecurrenceStack->addWidget(mRecurrence,0);

  mRecurrenceDisabled = new QLabel(
      i18n("This event does not recur.\nEnable Recurrence in General Tab."),
      mRecurrenceStack);
  mRecurrenceDisabled->setAlignment(AlignCenter);
  mRecurrenceStack->addWidget(mRecurrenceDisabled,1);
}

void KOEventEditor::enableRecurrence(bool enable)
{
  if (enable) mRecurrenceStack->raiseWidget(mRecurrence);
  else mRecurrenceStack->raiseWidget(mRecurrenceDisabled);

  mRecurrence->setEnabled(enable);
}

void KOEventEditor::editEvent(Event *event)
{
  init();

  mEvent = event;
  readEvent(mEvent);
}

void KOEventEditor::newEvent( QDateTime from, QDateTime to, bool allDay )
{
  init();

  if( KOPrefs::instance()->mGroupwareCommunication )
    mDetails->insertMyself();

  mEvent = 0;
  setDefaults(from,to,allDay);
}

void KOEventEditor::loadDefaults()
{
    int fmt = KOPrefs::instance()->mStartTime;

  QDateTime from(QDate::currentDate(), QTime(fmt,0,0));
  QDateTime to(QDate::currentDate(),
               QTime(fmt+KOPrefs::instance()->mDefaultDuration,0,0));

  setDefaults(from,to,false);
}

bool KOEventEditor::processInput()
{
  if (!validateInput()) return false;

  if (mEvent) {
    bool rc = true;
    Event *event = mEvent->clone();
    writeEvent(event);

    if( *mEvent == *event )
      // Don't do anything
      kdDebug(5850) << "Event not changed\n";
    else {
      kdDebug(5850) << "Event changed\n";
      int revision = event->revision();
      event->setRevision( revision + 1 );
      if( !KOPrefs::instance()->mGroupwareCommunication ||
	  KOGroupware::instance()->sendICalMessage( this, KCal::Scheduler::Request, event ) )
      {
	// Accept the event changes
	writeEvent( mEvent );
	emit eventChanged(mEvent);
      } else {
	// Revert the changes
	event->setRevision( revision );
	rc = false;
      }
    }
    delete event;
    return rc;
  } else {
    Event *event = new Event;
    event->setOrganizer(KOPrefs::instance()->email());
    writeEvent(event);
    if( !KOPrefs::instance()->mGroupwareCommunication ||
	KOGroupware::instance()->sendICalMessage( this, KCal::Scheduler::Request, event ) ) {
      mCalendar->addEvent(event);
      mEvent = event;
      emit eventAdded(event);
    } else {
      delete event;
      return false;
    }
  }

  return true;
}

void KOEventEditor::deleteEvent()
{
  kdDebug(5850) << "Delete event" << endl;
  if (mEvent) {
    if (KOPrefs::instance()->mConfirm && (!KOPrefs::instance()->mGroupwareCommunication ||
					  KOPrefs::instance()->email() == mEvent->organizer()))
    {
      switch (msgItemDelete()) {
        case KMessageBox::Continue: // OK
          emit eventToBeDeleted(mEvent);
          emit dialogClose(mEvent);
          mCalendar->deleteEvent(mEvent);
          emit eventDeleted();
          reject();
          break;
      }
    }
    else {
      emit eventToBeDeleted(mEvent);
      emit dialogClose(mEvent);
      mCalendar->deleteEvent(mEvent);
      emit eventDeleted();
      reject();
    }
  } else {
    reject();
  }
}

void KOEventEditor::setDefaults(QDateTime from, QDateTime to, bool allDay)
{
  mGeneral->setDefaults(from,to,allDay);
  mDetails->setDefaults(from,to,allDay);
#ifndef KORG_NORECURRENCE
  mRecurrence->setDefaults(from,to,allDay);

  enableRecurrence(false);
#endif
}

void KOEventEditor::readEvent( Event *event, bool tmpl )
{
  mGeneral->readEvent( event, tmpl );
  mDetails->readEvent( event, tmpl );
#ifndef KORG_NORECURRENCE
  mRecurrence->readEvent( event );

  enableRecurrence( event->recurrence()->doesRecur() );
#endif

  // categories
  mCategoryDialog->setSelected( event->categories() );
}

void KOEventEditor::writeEvent(Event *event)
{
  mGeneral->writeEvent( event );
  mDetails->writeEvent( event );

  if ( event->organizer() == KOPrefs::instance()->email() ) {
    Event *ev = new Event( *event );
    ev->registerObserver(0);
    mDetails->cancelAttendeeEvent( ev );
    if ( ev->attendeeCount() > 0 ) {
      emit deleteAttendee( ev );
    }
    delete(ev);
  }

#ifndef KORG_NORECURRENCE
  mRecurrence->writeEvent(event);
#endif
}

bool KOEventEditor::validateInput()
{
  if (!mGeneral->validateInput()) return false;
  if (!mDetails->validateInput()) return false;
#ifndef KORG_NORECURRENCE
  if (!mRecurrence->validateInput()) return false;
#endif
  return true;
}

int KOEventEditor::msgItemDelete()
{
  return KMessageBox::warningContinueCancel(this,
      i18n("This item will be permanently deleted."),
      i18n("KOrganizer Confirmation"),i18n("Delete"));
}

void KOEventEditor::slotLoadTemplate()
{
  CalendarLocal cal;
  Event *event = new Event;
  QString templateName = loadTemplate( &cal, event->type(),
                                       KOPrefs::instance()->mEventTemplates );
  delete event;
  if ( templateName.isEmpty() ) {
    return;
  }

  QPtrList<Event> events = cal.events();
  event = events.first();
  if ( !event ) {
    KMessageBox::error( this,
        i18n("Template does not contain a valid Event.")
        .arg( templateName ) );
  } else {
    kdDebug(5850) << "KOEventEditor::slotLoadTemplate(): readTemplate" << endl;
    readEvent( event, true );
  }
}

void KOEventEditor::slotSaveTemplate()
{
  createSaveTemplateDialog( SaveTemplateDialog::EventType );
}

void KOEventEditor::saveTemplate( const QString &templateName )
{
  Event *event = new Event;
  writeEvent( event );
  saveAsTemplate( event, templateName );
}

void KOEventEditor::emitCloseSignal()
{
  emit closingEventDialog( this );
}
