/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/********************** floating point dequantisation **************************/

/*  dequantize                                                               */
/*                                                                           */
/*  Description:                                                             */
/*    Dequantize and premultiply a block                                     */
/*                                                                           */
/*  Arguments:                                                               */
/*    short *block: a 8x8 block of integer                                   */
/*    dct_t *cache: the resulting 8x8 prescaled block of dct_t               */
/*    dct_t *dqmatrix: 8x8 dequantisation matrix                             */
/*    dct_t *psmatrix: 8x8 prescale coefficients                             */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */
/*                                                                           */
/*  Notes :                                                                  */
/*    Unfortunately mismatch control is different in MPEG-1 and MPEG-4 :(    */

static void inline dequantize_intra_global(short *block,
					   dct_t *cache,
					   dct_t *dqmatrix,
					   dct_t *psmatrix,
					   dct_t *mismatch /* not used */)
{
  int i, s, c;

  /* dequantize */
  c = (int) (block[0] * dqmatrix[0]);
  s = c;
  cache[0] = c;

  for(i = 1; i < 64; i++) {
    c = ((int) (block[i] * dqmatrix[i])) / 8;
    s ^= c;
    cache[i] = c;
  }

  /* mismatch control */
  if (!(s & 1)) {
    c ^= 1;
    cache[63] = c;
  }

  /* prescale for iDCT */
  for(i = 0; i < 64; i++)
    cache[i] *= psmatrix[i];
}

static void inline dequantize_intra_local(short *block,
					  dct_t *cache,
					  dct_t *dqmatrix,
					  dct_t *psmatrix,
					  dct_t *mismatch /* not used */)
{
  int i, c;

  /* dequantize */
  c = (int) (block[0] * dqmatrix[0]);
  cache[0] = c;

  for(i = 1; i < 64; i++) {
    c = ((int) (block[i] * dqmatrix[i])) / 8;
    /* mismatch control */
    if(block[i] > 0)
      c = (c - 1) | 1;
    else
      c |= 1;
    cache[i] = c;
  }

  /* prescale for iDCT */
  for(i = 0; i < 64; i++)
    cache[i] *= psmatrix[i];
}

/*  dequantize_inter                                                         */
/*                                                                           */
/*  Description:                                                             */
/*    Dequantize and premultiply a block for inter blocks                    */
/*    These two steps have to be done separetely for inter blocks :(         */
/*                                                                           */
/*  Arguments:                                                               */
/*    short *block: a 8x8 block of integer                                   */
/*    dct_t *cache: the resulting 8x8 prescaled block of dct_t               */
/*    dct_t *dqmatrix: 8x8 dequantisation matrix                             */
/*    dct_t *psmatrix: 8x8 prescale coefficients                             */
/*                                                                           */
/*  Return value:                                                            */
/*    None.                                                                  */
/*                                                                           */
/*  Notes :                                                                  */
/*    Unfortunately mismatch control is different in MPEG-1 and MPEG-4 :(    */

static void inline dequantize_inter_global(short *block,
					   dct_t *cache,
					   dct_t *dqmatrix,
					   dct_t *psmatrix,
					   dct_t *mismatch /* not used */)
{
  int i, s, c;

 /* dequantize */
  s = 0;
  for(i = 0; i < 64; i++) {
    c = 0;

    if(block[i] > 0)
      c = ((int) ((2*block[i]+1) * dqmatrix[i])) / 16;
    if(block[i] < 0)
      c = ((int) ((2*block[i]-1) * dqmatrix[i])) / 16;
    s ^= c;
    cache[i] = c;
  }

  /* mismatch control */
  if (!(s & 1)) {
    c ^= 1;
    cache[63] = c;
  }

  /* prescale for iDCT */
  for(i = 0; i < 64; i++)
    cache[i] *= psmatrix[i];
}

static void inline dequantize_inter_local(short *block,
					  dct_t *cache,
					  dct_t *dqmatrix,
					  dct_t *psmatrix,
					  dct_t *mismatch /* not used */)
{
  int i, c;

  /* dequantize */
  for(i = 0; i < 64; i++) {
    c = 0;

    if(block[i] > 0) {
      c = ((int) ((2*block[i]+1) * dqmatrix[i])) / 16;
      /* mismatch control */
      c = (c - 1) | 1;
    }
    if(block[i] < 0) {
      c = ((int) ((2*block[i]-1) * dqmatrix[i])) / 16;
      /* mismatch control */
      c |= 1;
    }
    cache[i] = c;
  }

  /* prescale for iDCT */
  for(i = 0; i < 64; i++)
    cache[i] *= psmatrix[i];
}

