/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*********************** compute mean of pixels in block *********************/
#ifndef __MEAN_INT_H__
#define __MEAN_INT_H__

/*  mean_withoutmask                                                         */
/*                                                                           */
/*  Description:                                                             */
/*    Compute mean of pixels values in a macroblock.                         */
/*                                                                           */
/*  Arguments:                                                               */
/*    unsigned char *input: the input macroblock                             */
/*    short pitch: number of pixels to the next line                         */
/*    unsigned short *mean: the returned mean.                               */
/*                                                                           */
/*  Return value:                                                            */
/*    unsigned short : pixel count.                                          */

static unsigned short inline mean_withoutmask(unsigned char *input,
					      short pitch,
					      unsigned long *mean)
{
  int i, j;
  unsigned long s;

  s = 0;
  for(i = 0; i < 8; i++) {
    for(j = 0; j < 8; j++)
	s += input[j];
    input += pitch;
  }

  *mean = (s >> 6);

  return(64);
}

/*  mean_withmask                                                            */
/*                                                                           */
/*  Description:                                                             */
/*    Compute mean of pixels values in an macroblock using a binary mask.    */
/*                                                                           */
/*  Arguments:                                                               */
/*    unsigned char *input: the input macroblock                             */
/*    unsigned char *mask: the input binary alpha block                      */
/*    short pitch: number of pixels to the next line                         */
/*    unsigned short *mean: the returned mean.                               */
/*                                                                           */
/*  Return value:                                                            */
/*    unsigned short : pixel count.                                          */

static unsigned short inline mean_withmask(unsigned char *input,
					   unsigned char *mask,
					   short pitch,
					   unsigned long *mean)
{
  int i, j;
  unsigned short n;
  unsigned long s;

  n = 0;
  s = 0;
  for(i = 0; i < 8; i++) {
    for(j = 0; j < 8; j++) {
      if(mask[j]) {
	s += input[j];
	n++;
      }
    }
    input += pitch;
    mask += pitch;
  }
  
  if(n == 0) { 
    *mean = 0;
    return(0);
  }

  *mean = s / n;

  return(n);
}
#endif
